/**
 * 
 */
package org.gcube.data.gml.elements;

import static org.gcube.data.gml.constants.Labels.*;
import static org.gcube.data.gml.elements.Conversions.*;
import static org.gcube.data.tml.uri.URIs.*;
import static org.gcube.data.trees.Constants.*;
import static org.gcube.data.trees.io.Bindings.*;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;

import org.gcube.data.trees.data.Edge;
import org.gcube.data.trees.data.InnerNode;
import org.gcube.data.trees.data.Leaf;
import org.gcube.data.trees.data.Node;
import org.gcube.data.trees.data.Tree;


/**
 * A gCube Document.
 * 
 * <p>
 * 
 * This implementation if thread-unsafe. Where required, thread-safety is responsibility of clients.
 *  
 * @author Federico De Faveri defaveri@isti.cnr.it
 * @author Lucio Lelii lucio.lelii@isti.cnr.it	
 * @author Fabio Simeoni (University of Strathclyde)
 */
@XmlRootElement(name=ROOT_NAME,namespace=TREE_NS)
public class GCubeDocument extends BaseElement {

	@XmlAttribute(name=COLLID_ATTR,namespace=TREE_NS) 
	private String collID;

	@XmlElement(name=PART)
	private List<GCubePart> parts = new LinkedList<GCubePart>();

	@XmlElement(name=ALTERNATIVE) 
	private List<GCubeAlternative> alternatives = new LinkedList<GCubeAlternative>();

	@XmlElement(name=METADATA) 
	private List<GCubeMetadata> metadata = new LinkedList<GCubeMetadata>();

	@XmlElement(name=ANNOTATION) 
	private List<GCubeAnnotation> annotations = new LinkedList<GCubeAnnotation>();


	private MetadataElements metadataElements = new MetadataElements(this, metadata);
	private AnnotationElements annotationElements = new AnnotationElements(this,annotations);
	private InnerElements<GCubePart> partElements = new InnerElements<GCubePart>(this,parts);
	private InnerElements<GCubeAlternative> alternativeElements = new InnerElements<GCubeAlternative>(this,alternatives);

	
	private Tree source;
	
	private boolean tracking = false;
	
	/**
	 * Creates an instance.
	 */
	public GCubeDocument() {} //used for deserialisation, and add use cases
	
	
	/**
	 * Creates an instance that act as a proxy for a given document.
	 * @param id the document identifier.
	 */
	public GCubeDocument(String id) {//used for staging
		super(id);
	}
	
	/**
	 * Creates an instance that act as a proxy for a given document in  given collection.
	 * @param id the document identifier.
	 * @param collectionID the collection identifier
	 */
	public GCubeDocument(String id, String collectionID) {//used for staging
		this(id);
		setCollectionID(collectionID);
	}
	
	
	 /**
	  * Returns the identifier of the document's collection
	  * @return the identifier.
	  */
	 public String collectionID() {
		 return collID;
	 }
	 
	 /**
	  * Sets the identifier of the document's collection.
	  * @param id the collection identifier
	  * @throws IllegalArgumentException if the collection identifier is <code>null</code>.
	  * @throws IllegalStateException if the document has already a collection identifier.
	  */
	 public void setCollectionID(String id) throws IllegalArgumentException,IllegalStateException {
		 
		 if (collID!=null)
			 throw new IllegalStateException("document is already bound to collection "+collID);
		 
		 if (id==null)
			 throw new IllegalArgumentException("collection identifier is "+id);
		 collID = id;
	 }
	 
	 /**{@inheritDoc}*/
	public URI uri() throws IllegalStateException, URISyntaxException {
		
		if (isNew())
			throw new IllegalStateException("new documents have no URI");
		
		if (collectionID()==null)
			throw new IllegalStateException("documents in no collection have no URI");
			
		return make(collectionID(),id());
			
	}
	
	
	/** 
	 * Returns the {@link GCubeMetadata} elements of the document for read and write access.
	 * @return the elements.
	 */
	public MetadataElements metadata() {
		return metadataElements;
	}
	
	/**
	 * Returns the {@link GCubeAnnotation} elements of the document for read and write access.
	 * @return the elements.
	 */
	public AnnotationElements annotations() {
		return annotationElements;
	}
	
	/**
	 * Returns the {@link GCubePart} elements of the document for read and write access.
	 * @return the elements.
	 */
	public InnerElements<GCubePart> parts() {
		return partElements;
	}
	
	/**
	 * Returns the {@link GCubeAlternative} elements of the document for read and write access.
	 * @return the elements.
	 */
	public InnerElements<GCubeAlternative> alternatives() {
		return alternativeElements;
	}
	
	/**
	 * Returns all the elements of the document, in no particular order.
	 * @return the documents;
	 */
	public Map<String,BaseInnerElement> elements() {
		Map<String,BaseInnerElement> elements = new HashMap<String,BaseInnerElement>();
		for (BaseInnerElement e : metadata) elements.put(e.id(),e);
		for (BaseInnerElement e : annotations) elements.put(e.id(),e);
		for (BaseInnerElement e : alternatives) elements.put(e.id(),e);
		for (BaseInnerElement e : parts) elements.put(e.id(),e);
		return elements;
	}
	
	//used internally to notify document that is has been bound from a serialisation
	 void postBinding() throws Exception {
		
		if (id()==null)
			throw new IllegalStateException(this+" has no identifier");
		
		if (collectionID()==null)
			throw new IllegalStateException(this+" is in no collection");
		
		//binds elements to self
		for (BaseInnerElement e : elements().values())
			e.postBinding(this);
		
		//a parsed document is immediately tracked for changes
		trackChanges();

	}
	
	public boolean isTracked() {
		return tracking;
	}
	
	
	public void trackChanges() throws IllegalStateException, Exception {
		
		if (isNew())
			throw new IllegalArgumentException("cannot track changes on a new document");
		
		if (isTracked())
			throw new IllegalStateException("document is already tracked for changes");
		
		//generate tree to measure updates from
		source = toTree(this);
		
		tracking=true;
	}
	
	public void resetChanges() throws IllegalStateException {
		
		if (!isTracked())
			throw new IllegalStateException("document is not currently tracked for changes");
		
		tracking=false;
	}
	

	public Tree delta() throws IllegalStateException, Exception {
		
		if (!isTracked())
			throw new IllegalStateException("document is not tracked for changes");
		
		Tree current = toTree(this);
		
		//add identifiers to enable delta
		toogleLeafIds(source);
		toogleLeafIds(current);
		
		//compute delta
		Tree delta = source.delta(current);
		
		//remove identifiers from source and delta
		toogleLeafIds(source);
		toogleLeafIds(delta);
		
		return delta;
		
	}
	
	//helper: adds ids on leaves if there aren't. removes them if there are
	private void toogleLeafIds(Node n) {
		
		if (n instanceof InnerNode)
			for (Edge e : ((InnerNode)n).edges()) {
				Node child = e.target();
				if (child instanceof Leaf) {
					Leaf newleaf = new Leaf(child.id()==null?e.label().getLocalPart():null, child.state(), ((Leaf) child).value(), child.attributes());
					e.target(newleaf);
				}
				else
					toogleLeafIds(child);
			}		
	}

	 //for testing
	Tree source() {
		return source;
    }
	 
	
	/** {@inheritDoc} */ 
	@Override
	public String toString() {
		
		StringBuilder builder = new StringBuilder(super.toString());
		builder.append(", parts=");
		builder.append(parts);
		builder.append(", alternatives=");
		builder.append(alternatives);
		builder.append(", metadata=");
		builder.append(metadata);
		builder.append(", annotations=");
		builder.append(annotations);
		return builder.toString();
	}
	
	
	
	





	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		result = prime * result
				+ ((alternatives == null) ? 0 : alternatives.hashCode());
		result = prime * result
				+ ((annotations == null) ? 0 : annotations.hashCode());
		result = prime * result + ((collID == null) ? 0 : collID.hashCode());
		result = prime * result
				+ ((metadata == null) ? 0 : metadata.hashCode());
		result = prime * result + ((parts == null) ? 0 : parts.hashCode());
		return result;
	}


	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (!super.equals(obj))
			return false;
		if (!(obj instanceof GCubeDocument))
			return false;
		GCubeDocument other = (GCubeDocument) obj;
		if (alternatives == null) {
			if (other.alternatives != null)
				return false;
		} else if (!alternatives.equals(other.alternatives))
			return false;
		if (annotations == null) {
			if (other.annotations != null)
				return false;
		} else if (!annotations.equals(other.annotations))
			return false;
		if (collID == null) {
			if (other.collID != null)
				return false;
		} else if (!collID.equals(other.collID))
			return false;
		if (metadata == null) {
			if (other.metadata != null)
				return false;
		} else if (!metadata.equals(other.metadata))
			return false;
		if (parts == null) {
			if (other.parts != null)
				return false;
		} else if (!parts.equals(other.parts))
			return false;
		return true;
	}
	
	
	
}
