/**
 * 
 */
package org.gcube.data.gml.elements;

import static eu.medsea.mimeutil.MimeUtil.*;
import static org.apache.axis.encoding.Base64.*;
import static org.apache.commons.io.IOUtils.*;
import static org.gcube.data.gml.constants.Labels.*;
import static org.gcube.data.gml.elements.Conversions.*;
import static org.gcube.data.trees.io.Bindings.*;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.net.URI;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.data.trees.Constants;

import eu.medsea.mimeutil.MimeException;
import eu.medsea.mimeutil.MimeType;
import eu.medsea.mimeutil.MimeUtil;

/**
 * Partial implementation of {@link GCubeElement}s.
 * The implementation is thread-unsafe. Where required, thread-safety is responsibility of clients.
 * 
 * @author Federico De Faveri defaveri@isti.cnr.it
 * @author Fabio Simeoni (University of Strathclyde)
 *
 */
public abstract class BaseElement implements GCubeElement {

	private GCUBELog logger = new GCUBELog(this);

	static {
		MimeUtil.registerMimeDetector("eu.medsea.mimeutil.detector.MagicMimeMimeDetector");
	}

	@XmlAttribute(name=ID_ATTR,namespace=Constants.TREE_NS) 
	private String id;

	@XmlElement(name=NAME) 
	private String name;

	@XmlElement(name=TYPE) 
	private String type;
	
	@XmlElement(name=CREATION_TIME)
	private Calendar creationTime;

	@XmlElement(name=LAST_UPDATE)
	private Calendar lastUpdateTime;

	@XmlElement(name=MIME_TYPE)
	private String mimeType;

	@XmlElement(name=BYTESTREAM_URI) 
	private URI bytestreamURI;

	@XmlElement(name=BYTESTREAM) 
	private String content;

	@XmlElement(name=LENGTH)
	private Long length;

	@XmlElement(name=SCHEMA_URI) 
	private URI schemaURI;

	@XmlElement(name=LANGUAGE) 
	private String language;

	@XmlElement(name=SCHEMA_NAME) 
	private String schemaName;

	@XmlElement(name=PROPERTY)
	private List<GCubeElementProperty> properties = new ArrayList<GCubeElementProperty>();


	BaseElement() {} //creates a new element

	BaseElement(String elementID) {//creates an element proxy
		id = elementID;
	}

	/**
	 * Indicates whether the element is new within the system or if it proxies instead an existing element.
	 * @return <code>true</code> if the element is new, <code>false</code> if it is a proxy.
	 */
	public boolean isNew() {
		return id()==null;
	}


	/** {@inheritDoc}*/
	public String id() {
		return id;
	}

	/** {@inheritDoc}*/
	public Calendar creationTime() { //read-only property set by the repository upon adding
		return creationTime;
	}

	/** {@inheritDoc}*/
	public Calendar lastUpdate() { //read-only property set by the repository upon adding/updating
		return lastUpdateTime;
	}

	/**{@inheritDoc}*/
	public String name() {
		return name;
	}

	/**
	 * Sets the element name.
	 * @param n the name.
	 */
	public void setName(String n) {
		name = n;
	}

	/**{@inheritDoc}*/
	public String type() {
		return type;
	}

	/**
	 * Sets the element type.
	 * @param t the type.
	 */
	public void setType(String t) {
		type = t;
	}


	/** {@inheritDoc}*/
	public String mimeType() {
		return mimeType;
	}

	/**
	 * Sets the content's Mime type.
	 * @param type the Mime type.
	 */
	public void setMimeType(String type) throws IllegalArgumentException {		
		mimeType = type;
	}


	/** {@inheritDoc}*/
	public URI bytestreamURI() {
		return bytestreamURI;
	}

	/**
	 * Sets the URI of the bytestream.
	 * @param uri the URI.
	 */
	public void setBytestreamURI(URI uri) {
		this.bytestreamURI = uri;
	}

	private void detectMimeType() {

		try {

			MimeType m = getMostSpecificMimeType(getMimeTypes(bytestream()));
			mimeType = m.toString();

		}
		catch(MimeException me) {
			mimeType = null;
		}

	}
	
	/** {@inheritDoc}*/
	public InputStream resolveBytestream() throws IOException {
		
		//bytestream available by-ref
		URI uri = bytestreamURI(); 
		if (uri!=null) {
			logger.trace("resolving "+uri);
			return uri.toURL().openStream();
		}
		//bytestream inlined
		byte[] stream = bytestream();
		if (stream!=null)
				return new ByteArrayInputStream(stream);
		
		//no bytestream available
		return null;
			
	}

	/** {@inheritDoc}*/
	public byte[] bytestream() {
		return content==null?null:decode((content));
	}

	/**
	 * Sets the element's bytestream.
	 * @param stream an {@link InputStream} from which the bytestream can be read, 
	 * or <code>null</code> to overwrite the current value.
	 * <br>
	 * Note the stream will be closed when the operation is completed.
	 * @throws IOException if the stream could not be processed.
	 */
	public void setBytestream(InputStream stream) throws IOException {

		if (stream==null){ 
			//clients may well want to do this
			content=null;
		} else {

			ByteArrayOutputStream bos = new ByteArrayOutputStream();

			copy(stream, bos);

			setBytestream(bos.toByteArray());

			try {
				bos.close();
				stream.close();
			}
			catch(Exception e) {
				logger.error("could not close temporary bytestream ",e);
			}
		}
	}

	/**
	 * Sets the element's bytestream.
	 * @param stream the stream, or <code>null</code> to unset the current value.
	 * <br>
	 * Note the stream will be closed when the operation is completed.
	 * @throws IOException if the stream could not be processed.
	 */
	public void setBytestream(Reader stream) throws IOException {

		if (stream==null) { //clients may well want to do this
			content=null;
		} else {

			ByteArrayOutputStream bos = new ByteArrayOutputStream();

			copy(stream, bos, "UTF8");

			setBytestream(bos.toByteArray());

			try {
				bos.close();
				stream.close();
			}
			catch(Exception e) {
				logger.error("could not close streams ",e);
			}
		}
	}


	/**
	 * Sets the element's bytestream.
	 * @param bytes the bytes of the stream, or <code>null</code> to unset the current value.
	 */
	public void setBytestream(byte[] bytes) {

		if (bytes==null) {
			content = null;
		} else {

			content = encode(bytes); //encode as text for XML-based network transmission

			length = (long) bytes.length;

			if (mimeType()==null) //detect if not already set
				detectMimeType();
		}
	}

	/** {@inheritDoc}*/ 
	public Long length() {
		return length;
	}
	
	
	/**
	 * Sets the length
	 * @param length the length
	 */
	public void setLenght(Long length)
	{
		this.length = length;
	}


	/** {@inheritDoc}*/
	public String language() {
		return language;
	}

	/**
	 * Sets the language of the element from the corresponding {@link Locale}.
	 * @param locale the {@link Locale} corresponding to the language.
	 */
	public void setLanguage(Locale locale) {

		language = locale==null? null : locale.getLanguage();
	}

	/**{@inheritDoc}*/
	public URI schemaURI() {
		return schemaURI;
	}

	/**
	 * Sets the URI of the element's schema.
	 * @param uri the URI.
	 */
	public void setSchemaURI(URI uri) {
		schemaURI = uri;
	}

	/**{@inheritDoc}*/
	public String schemaName() {
		return schemaName;
	}

	/**
	 * Sets the descriptive name of the element's schema.
	 * @param name the name.
	 */
	public void setSchemaName(String name) {
		this.schemaName = name;
	}


	/** {@inheritDoc}*/
	public Map<String, GCubeElementProperty> properties() {

		Map<String, GCubeElementProperty> map = new LinkedHashMap<String, GCubeElementProperty>();
		for (GCubeElementProperty p : properties) 
			map.put(p.key(), p);

		return map;
	}

	/** {@inheritDoc}*/ 
	public GCubeElementProperty property(String key) {
		for (GCubeElementProperty p : properties)
			if (p.key().equals(key))
				return p;
		return null;
	}

	/**
	 * Adds a generic property to the element, replacing any property with the same key.
	 * @param property the property.
	 * @return the property that this property replaces, or <code>null</code> if no such property exists.
	 */
	public GCubeElementProperty addProperty(GCubeElementProperty property) {
		GCubeElementProperty old = removeProperty(property.key());
		properties.add(property);
		return old;
	}

	/**
	 * Remove a generic property of the element.
	 * @param key the property key.
	 * @return the removed property, or <code>null</code> if the element has no property with the given key.
	 */
	public GCubeElementProperty removeProperty(String key) {
		GCubeElementProperty old = null;
		for (GCubeElementProperty p : properties)
			if (p.key().equals(key)) {
				old=p;
				break;
			}
		properties.remove(old);
		return old;
	}


	/**{@inheritDoc}*/ 
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("["+this.getClass().getSimpleName()+"] id=");
		builder.append(id);
		builder.append(", name=");
		builder.append(name);
		builder.append(", creationTime=");
		builder.append(calendarToString(creationTime));
		builder.append(", lastUpdateTime=");
		builder.append(calendarToString(lastUpdateTime));
		builder.append(", mimeType=");
		builder.append(mimeType);
		builder.append(", length=");
		builder.append(length);
		builder.append(", bytestream=");
		if (content!=null) builder.append(content.length()>100?content.substring(0,99)+"...(and other "+content.length()+" chars)":content);
		else builder.append("null");
		builder.append(", bytestreamURI=");
		builder.append(bytestreamURI);
		builder.append(", language=");
		builder.append(language);
		builder.append(", schemaURI=");
		builder.append(schemaURI);
		builder.append(", schemaName=");
		builder.append(schemaName);
		for (GCubeElementProperty p : properties) 
			builder.append(", property="+p.toString());
		return builder.toString();
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((content == null) ? 0 : content.hashCode());
		result = prime * result
		+ ((bytestreamURI == null) ? 0 : bytestreamURI.hashCode());
		result = prime * result
		+ ((creationTime == null) ? 0 : creationTime.hashCode());
		result = prime * result + ((id == null) ? 0 : id.hashCode());
		result = prime * result
		+ ((language == null) ? 0 : language.hashCode());
		result = prime * result
		+ ((lastUpdateTime == null) ? 0 : lastUpdateTime.hashCode());
		result = prime * result + ((length == null) ? 0 : length.hashCode());
		result = prime * result
		+ ((mimeType == null) ? 0 : mimeType.hashCode());
		result = prime * result + ((name == null) ? 0 : name.hashCode());
		result = prime * result
		+ ((properties == null) ? 0 : properties.hashCode());
		result = prime * result
		+ ((schemaName == null) ? 0 : schemaName.hashCode());
		result = prime * result
		+ ((schemaURI == null) ? 0 : schemaURI.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (!(obj instanceof BaseElement))
			return false;
		BaseElement other = (BaseElement) obj;
		if (content == null) {
			if (other.content != null)
				return false;
		} else if (!content.equals(other.content))
			return false;
		if (bytestreamURI == null) {
			if (other.bytestreamURI != null)
				return false;
		} else if (!bytestreamURI.equals(other.bytestreamURI))
			return false;
		if (creationTime == null) {
			if (other.creationTime != null)
				return false;
		} else if (!creationTime.equals(other.creationTime))
			return false;
		if (id == null) {
			if (other.id != null)
				return false;
		} else if (!id.equals(other.id))
			return false;
		if (language == null) {
			if (other.language != null)
				return false;
		} else if (!language.equals(other.language))
			return false;
		if (lastUpdateTime == null) {
			if (other.lastUpdateTime != null)
				return false;
		} else if (!lastUpdateTime.equals(other.lastUpdateTime))
			return false;
		if (length == null) {
			if (other.length != null)
				return false;
		} else if (!length.equals(other.length))
			return false;
		if (mimeType == null) {
			if (other.mimeType != null)
				return false;
		} else if (!mimeType.equals(other.mimeType))
			return false;
		if (name == null) {
			if (other.name != null)
				return false;
		} else if (!name.equals(other.name))
			return false;
		if (properties == null) {
			if (other.properties != null)
				return false;
		} else if (!properties.equals(other.properties))
			return false;
		if (schemaName == null) {
			if (other.schemaName != null)
				return false;
		} else if (!schemaName.equals(other.schemaName))
			return false;
		if (schemaURI == null) {
			if (other.schemaURI != null)
				return false;
		} else if (!schemaURI.equals(other.schemaURI))
			return false;
		//uris do not partecipate as they are dynamically generated upon parsing
		return true;
	}
}
