package org.gcube.datatransformation.datatransformationservice;

import gr.uoa.di.madgik.commons.server.PortRange;
import gr.uoa.di.madgik.commons.server.TCPConnectionManager;
import gr.uoa.di.madgik.commons.server.TCPConnectionManagerConfig;
import gr.uoa.di.madgik.grs.proxy.tcp.TCPConnectionHandler;
import gr.uoa.di.madgik.grs.proxy.tcp.TCPStoreConnectionHandler;
import gr.uoa.di.madgik.rr.ResourceRegistry;

import java.util.ArrayList;
import java.util.List;

import org.gcube.common.core.contexts.GCUBEServiceContext;
import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.faults.GCUBEUnrecoverableException;
import org.gcube.common.core.porttypes.GCUBEPortType;
import org.gcube.common.core.security.GCUBESecurityManager;
import org.gcube.common.core.security.GCUBESecurityManagerImpl;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.datatransformation.datatransformationservice.stubs.*;
import org.gcube.datatransformation.datatransformationlibrary.DTSCore;
import org.gcube.datatransformation.datatransformationlibrary.datahandlers.ContentTypeEvaluator;
import org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataSink;
import org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataSource;
import org.gcube.datatransformation.datatransformationlibrary.datahandlers.IOHandler;
import org.gcube.datatransformation.datatransformationlibrary.model.ContentType;
import org.gcube.datatransformation.datatransformationlibrary.model.TransformationProgram;
import org.gcube.datatransformation.datatransformationlibrary.model.TransformationUnit;
import org.gcube.datatransformation.datatransformationlibrary.reports.ReportManager;
import org.gcube.datatransformation.datatransformationlibrary.security.DTSSManager;
import org.gcube.datatransformation.datatransformationlibrary.transformation.model.TransformationDescription;

/**
 * @author Dimitris Katris, NKUA
 *
 * {@link DataTransformationService} is the class implementing the WS-Interface of the {@link DataTransformationServicePortType} 
 */
public class DataTransformationService extends GCUBEPortType {
	
	/**
	 * Logs operations performed by {@link DataTransformationService} class
	 */
	protected final static GCUBELog log = new GCUBELog(DataTransformationService.class);
	
	/**
	 * This method is invoked by the {@link DTSContext} on ready event
	 * When the service is ready it is able to do all the required initialization steps
	 * e.g. find info from the IS
	 * 
	 * @throws Exception If an underlying component could not be initialized.
	 */
	protected static void init() throws Exception{
		
		log.info("Initializing Data Transformation Service...");
		log.info("Initializing Resource Registry");
		ResourceRegistry.startBridging();

		initgRS2();		
		
		GCUBESecurityManager smanager;
		try {
			smanager = new GCUBESecurityManagerImpl() {public boolean isSecurityEnabled() {return DTSContext.getContext().isSecurityEnabled();}};
			DTSSManager.setSecurityManager(smanager);
		} catch (Exception e) {
			log.error("Could not initialize the security managener", e);
			throw new Exception("Could not initialize the security managener");
		}
		
		try {
			DTSSManager.init(DTSContext.getContext());
		} catch (Exception e) {
			log.error("Could not initialize DTSSManager", e);
			throw new Exception("Could not initialize DTSCore");
		}
		try {
			IOHandler.init(null);
		} catch (Exception e) {
			log.error("Could not initialize IOHandler", e);
			throw new Exception("Could not initialize IOHandler");
		}
		new StatisticsUpdater();
	}

	private static void initgRS2() {
		Integer gRS2Port = ((Integer)DTSContext.getContext().getProperty(DTSContext.GRS2_PORT, true)).intValue();
		log.info("Setting gRS2 port to "+ gRS2Port);

		PortRange ports = new PortRange(gRS2Port, gRS2Port);
		List<PortRange> portranges = new ArrayList<PortRange>();
		portranges.add(ports);
		
		TCPConnectionManager.Init(new TCPConnectionManagerConfig(
				GHNContext.getContext().getHostname(), portranges, true));
		
		TCPConnectionManager.RegisterEntry(new TCPConnectionHandler());
		TCPConnectionManager.RegisterEntry(new TCPStoreConnectionHandler());
	}
	
	/**
	 * This method transforms the input data to the target {@link ContentType} and stores the results to the output. In this method the service by its self discovers the {@link TransformationProgram} to use.
	 * 
	 * @param params The input (type + value), the output, the target {@link ContentType} and a boolean that denotes if the reporting mechanism shall be enabled.
	 * @return The output value as well as the identifier of the report (if requested). 
	 * @throws GCUBEFault If an error in transformData method occurred.
	 */
	public TransformDataResponse transformData(TransformData params) throws GCUBEFault {

		initTransaction();
		
		org.gcube.datatransformation.datatransformationlibrary.model.ContentType targetContentType; 
		try {
			targetContentType = StubsToModelUtils.contentTypeFromStub(params.getTargetContentType());
		} catch (Exception e){
			log.error("Target content type not properly set", e);
			throw new GCUBEUnrecoverableException("Target content type not properly set", e).toFault();
		}
		
		/* Creating the report */
		try {
			ReportManager.initializeReport(params.isCreateReport());
		} catch (Exception e) {
			log.error("Could not create report but continuing with the transformation...",e);
		}
		
		/* Getting the sink - Getting the sink first is better because source may start downloading objects*/
//		DataSink sink = null;
//		try {
//			sink = IOHandler.getDataSink(StubsToModelUtils.outputFromStub(params.getOutput()));
//		} catch (Exception e){
//			releaseResources();
//			log.error("Could not create data sink", e);
//			throw new GCUBEUnrecoverableException("Could not create DataSink from the given Output", e).toFault();
//		}
		
		/* Getting the source */
		DataSource source;
		try {
			source = IOHandler.getDataSource(StubsToModelUtils.inputFromStub(params.getInput()));
		} catch (Exception e){
			log.error("Could not create data source", e);
			releaseResources();
//			try {sink.close();} catch (Exception e1) {}
			throw new GCUBEUnrecoverableException("Could not create DataSource from the given Input", e).toFault();
		}
		
		TransformationDescription tDesc = new TransformationDescription(StubsToModelUtils.inputFromStub(params.getInput()), StubsToModelUtils.outputFromStub(params.getOutput()));
		
		/* Getting the dCore Instance and performing the requested transformation */
		DTSCore dCore = DTSSManager.getDTSCore();
		try {
			dCore.initializeAdaptor(tDesc);
		} catch (Exception e) {
			releaseResources();
//			try {sink.close();} catch (Exception e1) {}
			log.error("Could not initialize workflow adaptor", e);
			throw new GCUBEUnrecoverableException("Could not initialize workflow adaptor for the Data Transformation", e).toFault();
		}
		try {
			dCore.transformData(source, targetContentType);
		} catch (Exception e) {
			releaseResources();
			log.error("Did not manage to perform the transformation", e);
			throw new GCUBEUnrecoverableException("Did not manage to perform the transformation", e).toFault();
		}
		
		/* The sink returns the output */
		TransformDataResponse response = new TransformDataResponse();
		String output = tDesc.getReturnedValue();
		log.debug("Returning output "+output);
		response.setOutput(output);
		if(ReportManager.isReportingEnabled()){
			try {
				response.setReportEPR(ReportManager.getReport().getReportEndpoint());
				log.debug("Report RS EPR: " + ReportManager.getReport().getReportEndpoint());
			} catch (Exception e) {
				log.error("Could not get the RS EPR of the Report, but continuing nevertheless...", e);
			}
		}
		return response;
	}
	
	/**
	 * This method transforms the input data to the target {@link ContentType} and stores the results to the output. The {@link TransformationProgram} which will be used by the service is indicated by the client.
	 * 
	 * @param params The input (type + value), the output, the target {@link ContentType}, the transformation program id and a boolean that denotes if the reporting mechanism shall be enabled.
	 * @return The output value as well as the identifier of the report (if requested). 
	 * @throws GCUBEFault If an error in transformDataWithTransformationProgram method occurred.
	 */
	public TransformDataWithTransformationProgramResponse transformDataWithTransformationProgram(TransformDataWithTransformationProgram params) throws GCUBEFault {

		initTransaction();
		
		/* Just checking the parameters */
		String transformationProgramID = params.getTPID();
		if(transformationProgramID==null || transformationProgramID.trim().length()==0){
			log.error("Transformation program ID not set");
			throw new GCUBEFault("Transformation program ID not set");
		}
		
		org.gcube.datatransformation.datatransformationlibrary.model.ContentType targetContentType; 
		try {
			targetContentType = StubsToModelUtils.contentTypeFromStub(params.getTargetContentType());
		} catch (Exception e){
			log.error("Target content type not properly set", e);
			throw new GCUBEUnrecoverableException("Target content type not properly set", e).toFault();
		}
		
		org.gcube.datatransformation.datatransformationlibrary.model.Parameter[] unboundProgramParameters=null;
		try {
			unboundProgramParameters = StubsToModelUtils.parametersFromStub(params.getTProgramUnboundParameters());
		} catch (Exception e){
			log.error("Undefined error when converting unbound parameters from stub to model parameter type", e);
		}
		
		/* Creating the report */
		try {
			ReportManager.initializeReport(params.isCreateReport());
		} catch (Exception e) {
			log.error("Could not create report but continuing with the transformation...",e);
		}

		/* Getting the sink */
//		DataSink sink = null;
//		try {
//			sink = IOHandler.getDataSink(StubsToModelUtils.outputFromStub(params.getOutput()));
//		} catch (Exception e){
//			releaseResources();
//			log.error("Could not create data sink", e);
//			throw new GCUBEUnrecoverableException("Could not create DataSink from the given Output", e).toFault();
//		}
		
		/* Getting the source */
		DataSource source;
		try {
			source = IOHandler.getDataSource(StubsToModelUtils.inputFromStub(params.getInput()));
		} catch (Exception e){
			releaseResources();
//			try {sink.close();} catch (Exception e1) {}
			log.error("Could not create data source", e);
			throw new GCUBEUnrecoverableException("Could not create DataSource from the given Input", e).toFault();
		}
		
		TransformationDescription tDesc = new TransformationDescription(StubsToModelUtils.inputFromStub(params.getInput()), StubsToModelUtils.outputFromStub(params.getOutput()));
		
		/* Getting the dCore Instance and performing the requested transformation */
		DTSCore dCore = DTSSManager.getDTSCore();
		try {
			dCore.initializeAdaptor(tDesc);
		} catch (Exception e) {
			releaseResources();
//			try {sink.close();} catch (Exception e1) {}
			log.error("Could not initialize workflow adaptor", e);
			throw new GCUBEUnrecoverableException("Could not initialize workflow adaptor for the Data Transformation", e).toFault();
		}
		
		try {
			dCore.transformDataWithTransformationProgram(source, transformationProgramID, unboundProgramParameters, targetContentType);
		} catch(Exception e) {
			releaseResources();
			log.error("Could not perform the transformation",e);
			throw new GCUBEUnrecoverableException("Could not perform the transformation", e).toFault();
		}
		
		/* The sink returns the output */
		TransformDataWithTransformationProgramResponse response = new TransformDataWithTransformationProgramResponse();
		String output = tDesc.getReturnedValue();
		log.debug("Returning output "+output);
		response.setOutput(output);
		if(ReportManager.isReportingEnabled()){
			try {
				response.setReportEPR(ReportManager.getReport().getReportEndpoint());
				log.debug("Report RS EPR: " + ReportManager.getReport().getReportEndpoint());
			} catch (Exception e) {
				log.error("Could not get the RS EPR of the Report, but continuing nevertheless...", e);
			}
		}
		return response;
	}
	
	/**
	 * This method transforms the input data to the target {@link ContentType} and stores the results to the output. The {@link TransformationUnit} which will be used by the service is indicated by the client.
	 * 
	 * @param The input (type + value), the output, the target {@link ContentType}, the transformation program, unit id and a boolean that denotes if the reporting mechanism shall be enabled.
	 * @return The output value as well as the identifier of the report (if requested). 
	 * @throws GCUBEFault If an error in transformDataWithTransformationUnit method occurred.
	 */
	public TransformDataWithTransformationUnitResponse transformDataWithTransformationUnit(TransformDataWithTransformationUnit params) throws GCUBEFault{
		
		initTransaction();
		
		/* Just checking the parameters */
		String TProgramID = params.getTPID();
		if(TProgramID==null || TProgramID.trim().length()==0){
			log.error("Transformation program ID not set");
			throw new GCUBEFault("Transformation program ID not set");
		}
		
		String transformationUnitID = params.getTransformationUnitID();
		if(transformationUnitID==null || transformationUnitID.trim().length()==0){
			log.error("Transformation unit ID not set");
			throw new GCUBEFault("Transformation unit ID not set");
		}
		
		org.gcube.datatransformation.datatransformationlibrary.model.ContentType targetContentType; 
		try {
			targetContentType = StubsToModelUtils.contentTypeFromStub(params.getTargetContentType());
		} catch (Exception e){
			log.error("Target content type not properly set", e);
			throw new GCUBEUnrecoverableException("Target content type not properly set", e).toFault();
		}
		
		org.gcube.datatransformation.datatransformationlibrary.model.Parameter[] unboundProgramParameters=null;
		try {
			unboundProgramParameters = StubsToModelUtils.parametersFromStub(params.getTProgramUnboundParameters());
		} catch (Exception e){
			log.error("Undefined error when converting unbound parameters from stub to model parameter type", e);
		}
		
		/* Creating the report */
		try {
			ReportManager.initializeReport(params.isCreateReport());
		} catch (Exception e) {
			log.error("Could not create report but continuing with the transformation...",e);
		}

		/* Getting the sink */
//		DataSink sink = null;
//		try {
//			sink = IOHandler.getDataSink(StubsToModelUtils.outputFromStub(params.getOutput()));
//		} catch (Exception e){
//			releaseResources();
//			log.error("Could not create data sink", e);
//			throw new GCUBEUnrecoverableException("Could not create DataSink from the given Output", e).toFault();
//		}
		
		/* Getting the sources */
//		ArrayList<DataSource> sources = null;
//		for(int i=0;i<params.getInputs().length;i++){
//			Input input = params.getInputs(i);
//			try {
//				DataSource source = IOHandler.getDataSource(StubsToModelUtils.inputFromStub(input));
//				sources.add(source);
//			} catch (Exception e){
//				releaseResources();
//				try {sink.close();} catch (Exception e1) {}
//				log.error("Could not create DataSource from Input $"+i, e);
//				throw new GCUBEUnrecoverableException("Could not create DataSource from Input $"+i, e).toFault();
//			}
//		}
		
		boolean filterSources = params.isFilterSources();
		
		TransformationDescription tDesc = new TransformationDescription(StubsToModelUtils.inputFromStub(params.getInputs(0)), StubsToModelUtils.outputFromStub(params.getOutput()));
		
		/* Getting the dCore Instance and performing the requested transformation */
		DTSCore dCore = DTSSManager.getDTSCore();
		try {
			dCore.initializeAdaptor(tDesc);
		} catch (Exception e) {
			releaseResources();
//			try {sink.close();} catch (Exception e1) {}
			log.error("Could not initialize workflow adaptor", e);
			throw new GCUBEUnrecoverableException("Could not initialize workflow adaptor for the Data Transformation", e).toFault();
		}
		
		try {
			dCore.transformDataWithTransformationUnit(TProgramID, transformationUnitID, unboundProgramParameters, targetContentType, filterSources);
		} catch (Exception e) {
			releaseResources();
			log.error("Could not perform the requested transformation WithKnownTPIDAndTransformationID", e);
			throw new GCUBEUnrecoverableException("Could not perform the requested transformation with transformation unit.", e).toFault();
		}
		
		/* The sink returns the output */
		TransformDataWithTransformationUnitResponse response = new TransformDataWithTransformationUnitResponse();
		String output = tDesc.getReturnedValue();
		log.debug("Returning output "+output);
		response.setOutput(output);
		if(ReportManager.isReportingEnabled()){
			try {
				response.setReportEPR(ReportManager.getReport().getReportEndpoint());
				log.debug("Report RS EPR: " + ReportManager.getReport().getReportEndpoint());
			} catch (Exception e) {
				log.error("Could not get the RS EPR of the Report, but continuing nevertheless...", e);
			}
		}
		return response;
	}
	
	/**
	 * Releases any resources which were used for the transformation.
	 */
	private void releaseResources(){
		if(ReportManager.isReportingEnabled()){
			try {
				ReportManager.closeReport();
			} catch (Exception e) {
				log.error("Could not close report", e);
			}
		}
	}
	
	/**
	 * Searches for {@link TransformationUnit}s that are able to perform a transformation from a source to a target {@link ContentType}.
	 * 
	 * @param params A source and a target {@link ContentType}.
	 * @return One or more available transformation units that can perform the conversion from the source to the target {@link ContentType}.
	 * @throws GCUBEFault If an error in searching for transformation units occurred.
	 */
	public FindApplicableTransformationUnitsResponse findApplicableTransformationUnits(FindApplicableTransformationUnits params) throws GCUBEFault{
		
		initTransaction();
		
		org.gcube.datatransformation.datatransformationlibrary.model.ContentType sourceContentType; 
		try {
			sourceContentType = StubsToModelUtils.contentTypeFromStub(params.getSourceContentType());
		} catch (Exception e){
			log.error("Source content type not properly set", e);
			throw new GCUBEUnrecoverableException("Source content type not properly set", e).toFault();
		}
		
		org.gcube.datatransformation.datatransformationlibrary.model.ContentType targetContentType; 
		try {
			targetContentType = StubsToModelUtils.contentTypeFromStub(params.getTargetContentType());
		} catch (Exception e){
			log.error("Target content type not properly set", e);
			throw new GCUBEUnrecoverableException("Target content type not properly set", e).toFault();
		}
		
		boolean createAndPublishCompositeTP = params.isCreateAndPublishCompositeTP();
		
		DTSCore dCore = DTSSManager.getDTSCore();
		ArrayList<TransformationUnit> transformations;
		try {
			transformations = dCore.findApplicableTransformationUnits(sourceContentType, targetContentType, createAndPublishCompositeTP);
		} catch (Exception e) {
			log.error("Did not manage to search for applicable transformation units", e);
			throw new GCUBEUnrecoverableException("Did not manage to search for applicable transformation units", e).toFault();
		}
		
		FindApplicableTransformationUnitsResponse response = new FindApplicableTransformationUnitsResponse();
		if(transformations!=null && transformations.size()>0){
			ArrayList<TPAndTransformationUnit> TPAndTransformationUnitList = new ArrayList<TPAndTransformationUnit>(); 
			for(TransformationUnit transformation: transformations){
				TPAndTransformationUnit tpandtrid = new TPAndTransformationUnit();
				tpandtrid.setTransformationUnitID(transformation.getId());
				tpandtrid.setTransformationProgramID(transformation.getTransformationProgram().getId());
				TPAndTransformationUnitList.add(tpandtrid);
			}
			response.setTPAndTransformationUnitIDs(TPAndTransformationUnitList.toArray(new TPAndTransformationUnit[TPAndTransformationUnitList.size()]));
		}
		return response;
	}
	
	/**
	 * Searches for {@link ContentType}s to which an object can be transformed.
	 * 
	 * @param request The source {@link ContentType}.
	 * @return One or more available target {@link ContentType}s.
	 * @throws GCUBEFault If an error in searching for available target {@link ContentType}s occurred.
	 */
	public FindAvailableTargetContentTypesResponse findAvailableTargetContentTypes(FindAvailableTargetContentTypes request) throws GCUBEFault {
		initTransaction();
		org.gcube.datatransformation.datatransformationlibrary.model.ContentType sourceContentType; 
		try {
			sourceContentType = StubsToModelUtils.contentTypeFromStub(request.getSourceContentType());
		} catch (Exception e){
			log.error("Source content type not properly set", e);
			throw new GCUBEUnrecoverableException("Source content type not properly set", e).toFault();
		}
		
		DTSCore dCore = DTSSManager.getDTSCore();
		ArrayList<org.gcube.datatransformation.datatransformationlibrary.model.ContentType> targetContentTypes;
		try {
			targetContentTypes = dCore.getAvailableTargetContentTypes(sourceContentType);
		} catch (Exception e) {
			log.error("Did not manage to search for available target content types", e);
			throw new GCUBEUnrecoverableException("Did not manage to search for available target content types", e).toFault();
		}
		FindAvailableTargetContentTypesResponse response = new FindAvailableTargetContentTypesResponse();
		response.setTargetContentTypes(StubsToModelUtils.contentTypeArrayToStub(targetContentTypes));
		return response;
		
	}
	
	/**
	 * Performs a query to get information about the transformation programs.
	 * 
	 * @param query The query about the transformation programs.
	 * @return The result of the query in xml format.
	 * @throws GCUBEFault If an error in performing the query occurred.
	 */
	public String queryTransformationPrograms(String query) throws GCUBEFault{
		
		initTransaction();
		
		DTSCore dCore = DTSSManager.getDTSCore();
		try {
			return dCore.getIManager().queryTransformationPrograms(query);
		} catch (Exception e) {
			log.error("Could not query transfomration programs", e);
			throw new GCUBEUnrecoverableException("Could not query transfomration programs", e).toFault();
		}
	}
	
	/**
	 * Uses a {@link ContentType} evaluator which is identified by its type in order to evaluate the {@link ContentType} of an object.
	 * 
	 * @param request The Data Element id and the type of the content type evaluator
	 * @return The {@link ContentType} of the data element.
	 * @throws GCUBEFault If an error in evaluating the {@link ContentType} occurred. 
	 */
	public ArrayOfDataElementIDandContentType evaluateContentTypeByDataElementID(EvaluateContentTypeByDataElementID request) throws GCUBEFault{
		log.debug("Invocation to evaluateContentTypeByDataElementID for evaluator type: "+request.getEvaluatorType());
		String[] dataElementIDs = request.getDataElementIDs();
		if(dataElementIDs==null || dataElementIDs.length==0){
			throw new GCUBEFault("Object IDs to evaluate their ContentType are not set");
		}
		initTransaction();
		ContentTypeEvaluator evaluator;
		try {
			evaluator = IOHandler.getContentTypeEvaluator(request.getEvaluatorType());
		} catch (Exception e) {
			log.error("Could not create ContentTypeEvaluator", e);
			throw new GCUBEUnrecoverableException("Could not create ContentTypeEvaluator", e).toFault();
		}
		
		ArrayList<DataElementIDandContentType> objectIDandContentTypeList = new ArrayList<DataElementIDandContentType>();
		log.debug("Going to get the content format for "+dataElementIDs.length+" objects");
		for(String dataElementID: dataElementIDs){
			try {
				DataElementIDandContentType dataElementIDandContentType = new DataElementIDandContentType();
				dataElementIDandContentType.setDataElementID(dataElementID);
				dataElementIDandContentType.setContentType(StubsToModelUtils.contentTypeToStub(evaluator.evaluateContentTypeOfDataElement(dataElementID)));
				objectIDandContentTypeList.add(dataElementIDandContentType);
			} catch (Exception e) {
				log.error("Could not get content format for object "+dataElementID, e);
			}
		}
		log.debug("Managed to find the content format for "+objectIDandContentTypeList.size()+" objects");
		ArrayOfDataElementIDandContentType response = new ArrayOfDataElementIDandContentType();
		response.setDataElementIDAndContentType(objectIDandContentTypeList.toArray(new DataElementIDandContentType[objectIDandContentTypeList.size()]));
		return response;
	}
	
	/**
	 * Sets the appropriate info in the context of the transformation.
	 */
	private static void initTransaction(){
		try {
			DTSSManager.useCredentials(DTSContext.getContext().getCallerCredentials());
		} catch (Exception e) {
			log.warn("Could not get Caller Credentials", e);
		}
		DTSSManager.setScope(DTSContext.getContext().getScope().toString());
	}
	
	/* (non-Javadoc)
	 * @see org.gcube.common.core.porttypes.GCUBEPortType#getServiceContext()
	 */
	@Override
	protected GCUBEServiceContext getServiceContext() {
		return DTSContext.getContext();
	}
}

