package org.gcube.datatransformation.datatransformationlibrary.programs.metadata.xslttransformer;

import java.io.StringReader;
import java.io.StringWriter;
import java.util.List;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Templates;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.gcube.datatransformation.datatransformationlibrary.dataelements.impl.StrDataElement;
import org.gcube.datatransformation.datatransformationlibrary.dataelements.DataElement;
import org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataSink;
import org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataSource;
import org.gcube.datatransformation.datatransformationlibrary.model.ContentType;
import org.gcube.datatransformation.datatransformationlibrary.model.Parameter;
import org.gcube.datatransformation.datatransformationlibrary.programs.Program;
import org.gcube.datatransformation.datatransformationlibrary.programs.metadata.util.XSLTRetriever;
import org.gcube.datatransformation.datatransformationlibrary.reports.ReportManager;
import org.gcube.datatransformation.datatransformationlibrary.reports.Record.Status;
import org.gcube.datatransformation.datatransformationlibrary.reports.Record.Type;
import org.gcube.datatransformation.datatransformationlibrary.security.DTSSManager;

/**
 * @author Dimitris Katris, NKUA
 * <p>
 * Uses an xslt in order to transform metadata.
 * </p>
 */
public class XSLT_Transformer implements Program{

	private static Logger log = LoggerFactory.getLogger(XSLT_Transformer.class);
	
	/**
	 * @see org.gcube.datatransformation.datatransformationlibrary.programs.Program#transform(java.util.List, java.util.List, org.gcube.datatransformation.datatransformationlibrary.model.ContentType, org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataSink)
	 * @param sources The <tt>DataSources</tt> from which the <tt>Program</tt> will get the <tt>DataElements</tt>.
	 * @param programParameters The parameters of the <tt>Program</tt> which are primarily set by the <tt>TransformationUnit</tt>.
	 * @param targetContentType The <tt>ContentType</tt> in which the source data will be transformed.
	 * @param sink The <tt>DataSink</tt> in which the <tt>Program</tt> will append the transformed <tt>DataElements</tt>.
	 * @throws Exception If the program is not capable to transform <tt>DataElements</tt>.
	 */
	public void transform(List<DataSource> sources, List<Parameter> programParameters, ContentType targetContentType, DataSink sink) throws Exception {
		String xsltID=null;

		if(programParameters==null || programParameters.size()==0){
			log.error("Program parameters do not contain xslt");
			throw new Exception("Program parameters do not contain xslt");
		}
		
		for(Parameter param: programParameters){
			if(param.getName().toLowerCase().equals("xslt")){
				xsltID=param.getValue();
				break;
			}
		}
		String xslt;
		if(xsltID!=null && xsltID.trim().length()>0){
			log.debug("Got XSLT ID: "+xsltID);
			try {
				xslt=XSLTRetriever.getXSLTFromIS(xsltID, DTSSManager.getScope());
			} catch (Exception e) {
				log.error("Did not manage to retrieve the XSLT with ID "+xsltID+", aborting transformation...");
				throw new Exception("Did not manage to retrieve the XSLT with ID "+xsltID);
			}
		}else{
			log.error("Program parameters do not contain xslt");
			throw new Exception("Program parameters do not contain xslt");
		}
		
		Templates compiledXSLT = null;
		try {
			TransformerFactory factory = TransformerFactory.newInstance();
			compiledXSLT = factory.newTemplates(new StreamSource(new StringReader(xslt)));
		} catch (Exception e) {
			log.error("Failed to compile the XSLT: " + xslt, e);
			throw new Exception("Failed to compile the XSLT");
		}
		transformByXSLT(sources, compiledXSLT, targetContentType, sink);
	}
	
	private void transformByXSLT(List<DataSource> sources, Templates compiledXSLT, ContentType targetContentType, DataSink sink) throws Exception{
		if(sources.size()!=1){
			throw new Exception("Elm2ElmProgram is only applicable for programs with one Input");
		}
		DataSource source = sources.get(0);
		while(source.hasNext()){
			log.debug("Source has next...");
			DataElement object = source.next();
			if(object!=null){
				DataElement transformedObject;
				try {
					log.debug("Got next object with id "+object.getId());
					transformedObject = transformDataElementByXSLT(object, compiledXSLT, targetContentType);
					if(transformedObject==null){
						log.warn("Got null transformed object");
						throw new NullPointerException();
					}
					transformedObject.setId(object.getId());
					log.debug("Got transformed object with id: "+transformedObject.getId()+" and content format: "+transformedObject.getContentType().toString()+", appending into the sink");
					ReportManager.manageRecord(object.getId(), "Data element with id "+object.getId()+" and content format "+object.getContentType().toString()+" " +
							"was transformed successfully to "+transformedObject.getContentType().toString(), Status.SUCCESSFUL, Type.TRANSFORMATION);
				} catch (Exception e) {
					log.error("Could not transform Data Element, continuing to next...",e);
					ReportManager.manageRecord(object.getId(), "Data element with id "+object.getId()+" and content format "+object.getContentType().toString()+" " +
							"could not be transformed to "+targetContentType.toString(), Status.FAILED, Type.TRANSFORMATION);
					continue;
				}
				sink.append(transformedObject);
				log.debug("Transformed object with id: "+transformedObject.getId()+", was appended successfully");
			}else{
				log.warn("Got null object from the data source");
			}
			
		}
		log.debug("Source does not have any objects left, closing the sink...");
		sink.close();
	}
	
	private DataElement transformDataElementByXSLT(DataElement sourceDataElement, Templates compiledXSLT, ContentType targetContentType) throws Exception {
		StrDataElement transformedElement = StrDataElement.getSinkDataElement(sourceDataElement);
		transformedElement.setContentType(targetContentType);
		transformedElement.setId(sourceDataElement.getId());
		
		StringWriter output = new StringWriter();
		try {
			Transformer t = compiledXSLT.newTransformer();
			t.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
			if(sourceDataElement instanceof StrDataElement){
				t.transform(new StreamSource(new StringReader(((StrDataElement)sourceDataElement).getStringContent())), new StreamResult(output));
			}else{
				t.transform(new StreamSource(sourceDataElement.getContent()), new StreamResult(output));
			}
		} catch (Exception e) {
			log.error("Failed to transform element with ID = " + sourceDataElement.getId());
			throw new Exception("Failed to transform element with ID = " + sourceDataElement.getId());
		}
		
		transformedElement.setContent(output.toString());
		return transformedElement;
	}
}
