package org.gcube.datatransformation.datatransformationlibrary.programs.metadata.indexfeed;

import java.io.StringReader;
import java.io.StringWriter;
import java.util.List;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Templates;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathFactory;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.gcube.datatransformation.datatransformationlibrary.dataelements.DataElement;
import org.gcube.datatransformation.datatransformationlibrary.dataelements.impl.StrDataElement;
import org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataSink;
import org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataSource;
import org.gcube.datatransformation.datatransformationlibrary.model.ContentType;
import org.gcube.datatransformation.datatransformationlibrary.model.Parameter;
import org.gcube.datatransformation.datatransformationlibrary.programs.Program;
import org.gcube.datatransformation.datatransformationlibrary.programs.metadata.util.XMLStringParser;
import org.gcube.datatransformation.datatransformationlibrary.programs.metadata.util.XSLTRetriever;
import org.gcube.datatransformation.datatransformationlibrary.reports.ReportManager;
import org.gcube.datatransformation.datatransformationlibrary.reports.Record.Status;
import org.gcube.datatransformation.datatransformationlibrary.reports.Record.Type;
import org.gcube.datatransformation.datatransformationlibrary.security.DTSSManager;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

/**
 * @author Dimitris Katris, NKUA
 * <p>
 * Creates forward rowsets.
 * </p>
 */
public class FwRowset_Transformer implements Program{

	private static Logger log = LoggerFactory.getLogger(FwRowset_Transformer.class);

	private XPath xpath = XPathFactory.newInstance().newXPath();
	private Transformer serializer = null;
	private DOMSource serializerSource = new DOMSource();

	/**
	 * @see org.gcube.datatransformation.datatransformationlibrary.programs.Program#transform(java.util.List, java.util.List, org.gcube.datatransformation.datatransformationlibrary.model.ContentType, org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataSink)
	 * @param sources The <tt>DataSources</tt> from which the <tt>Program</tt> will get the <tt>DataElements</tt>.
	 * @param programParameters The parameters of the <tt>Program</tt> which are primarily set by the <tt>TransformationUnit</tt>.
	 * @param targetContentType The <tt>ContentType</tt> in which the source data will be transformed.
	 * @param sink The <tt>DataSink</tt> in which the <tt>Program</tt> will append the transformed <tt>DataElements</tt>.
	 * @throws Exception If the program is not capable to transform <tt>DataElements</tt>.
	 */
	public void transform(List<DataSource> sources, List<Parameter> programParameters, ContentType targetContentType, DataSink sink) throws Exception {
		if(programParameters==null || programParameters.size()==0){
			log.error("Program parameters do not contain xslt");
			throw new Exception("Program parameters do not contain xslt");
		}

		//Finding xslt and indexType from the program parameters...
		String xsltID=null;

		for(Parameter param: programParameters){
			if(param.getName().equalsIgnoreCase("xslt")){
				xsltID=param.getValue();
			}
		}

		String xslt;
		if(xsltID!=null && xsltID.trim().length()>0){
			log.debug("Got XSLT ID: "+xsltID);
			try {
				xslt=XSLTRetriever.getXSLTFromIS(xsltID, DTSSManager.getScope());
			} catch (Exception e) {
				log.error("Did not manage to retrieve the XSLT with ID "+xsltID+", aborting transformation...");
				throw new Exception("Did not manage to retrieve the XSLT with ID "+xsltID, e);
			}
		}else{
			log.error("Program parameters do not contain xslt");
			throw new Exception("Program parameters do not contain xslt");
		}

		NodeList keys = null;

		Templates compiledXSLT = null;
		TransformerFactory factory = null;
		try {
			factory = TransformerFactory.newInstance();
			compiledXSLT = factory.newTemplates(new StreamSource(new StringReader(xslt)));
		} catch (Exception e) {
			log.error("Failed to compile the XSLT: " + xslt, e);
			throw new Exception("Failed to compile the XSLT", e);
		}

		/* Create a transformer that will be used for serializing the transformed elements */

		try {
			serializer = factory.newTransformer();
			serializer.setOutputProperty("omit-xml-declaration", "yes");
		} catch (Exception e) {
			log.error("Failed to create serializer.", e);
			throw new Exception("Failed to create serializer.", e);
		}

		/* Retrieve the keys description from the XSLT definition */
		boolean foundKeysDesc = true;
		try {
			keys = (NodeList) xpath.evaluate("//*[local-name()='variable']/self::node()[@name='keys']/key",
					new InputSource(new StringReader(xslt)), XPathConstants.NODESET);
		} catch (Exception e) {
			foundKeysDesc = false;
		}

		if (!foundKeysDesc || keys==null || keys.getLength()==0) {
			log.error("Unable to locate the 'keys' variable in the given XSLT." +
					"Make sure the parameter is defined like this:\n" +
			"<xsl:variable name=\"keys\"> <key><keyName/><keyXPath/></key> ... </xsl:param>");
			throw new Exception("Unable to locate the 'keys' variable in the given XSLT." +
					"Make sure the parameter is defined like this:\n" +
			"<xsl:variable name=\"keys\"> <key><keyName/><keyXPath/></key> ... </xsl:param>");
		}

		/* Parse the key descriptions */
		Object[][] keyDescs = null;
		try {
			keyDescs = new Object[keys.getLength()][];
			for (int i=0; i<keys.getLength(); i++) {
				Node n = keys.item(i);
				keyDescs[i] = new Object[2];
				keyDescs[i][0] =  ((Element) n).getElementsByTagName("keyName").item(0).getTextContent();
				keyDescs[i][1] =  xpath.compile(((Element) n).getElementsByTagName("keyXPath").item(0).getTextContent());
				log.debug("Xpath: " +((Element) n).getElementsByTagName("keyXPath").item(0).getTextContent());
			}
		} catch (Exception e) {
			log.error("Failed to parse and compile the key descriptions.", e);
			throw new Exception("Failed to parse and compile the key descriptions.", e);
		}

		transformByXSLT(sources, compiledXSLT, targetContentType, sink, keyDescs);
	}

	private void transformByXSLT(List<DataSource> sources, Templates compiledXSLT, ContentType targetContentType, DataSink sink, Object[][] keyDescs) throws Exception{
		if(sources.size()!=1){
			throw new Exception("Elm2ElmProgram is only applicable for programs with one Input");
		}
		DataSource source = sources.get(0);
		while(source.hasNext()){
			log.debug("Source has next...");
			DataElement object = source.next();
			if(object!=null){
				DataElement transformedObject;
				try {
					log.debug("Got next object with id "+object.getId());
					transformedObject = transformDataElementByXSLT(object, compiledXSLT, targetContentType, keyDescs);
					if(transformedObject==null){
						log.warn("Got null transformed object...");
						throw new Exception();
					}
					transformedObject.setId(object.getId());
					log.debug("Got transformed object with id: "+transformedObject.getId()+" and content format: "+transformedObject.getContentType().toString()+", appending into the sink");
					ReportManager.manageRecord(object.getId(), "Data element with id "+object.getId()+" and content format "+object.getContentType().toString()+" " +
							"was transformed successfully to "+transformedObject.getContentType().toString(), Status.SUCCESSFUL, Type.TRANSFORMATION);
				} catch (Exception e) {
					log.error("Could not transform Data Element, continuing to next...",e);
					ReportManager.manageRecord(object.getId(), "Data element with id "+object.getId()+" and content format "+object.getContentType().toString()+" " +
							"could not be transformed to "+targetContentType.toString(), Status.FAILED, Type.TRANSFORMATION);
					continue;
				}
				sink.append(transformedObject);
				log.debug("Transformed object with id: "+transformedObject.getId()+", was appended successfully");
			}else{
				log.warn("Got null object from the data source");
			}

		}
		log.debug("Source does not have any objects left, closing the sink...");
		sink.close();
	}

	private DataElement transformDataElementByXSLT(DataElement sourceDataElement, Templates compiledXSLT,
			ContentType targetContentType, Object[][] keyDescs) throws Exception {

		StrDataElement transformedElement = StrDataElement.getSinkDataElement(sourceDataElement);
		transformedElement.setContentType(targetContentType);
		transformedElement.setId(sourceDataElement.getId());

		/* Transform the current source element using the given XSLT. The output of the XSLT
		 * is a forward index rowset whose 'key' is empty, because the actual key is filled in
		 * by this program later.
		 */
		StringWriter output = new StringWriter();
		Transformer t = compiledXSLT.newTransformer();
		t.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
		if(sourceDataElement instanceof StrDataElement){
			t.transform(new StreamSource(new StringReader(((StrDataElement)sourceDataElement).getStringContent())), new StreamResult(output));
		}else{
			t.transform(new StreamSource(sourceDataElement.getContent()), new StreamResult(output));
		}
		String transformedPayload = output.toString();
		Document transformedDoc = XMLStringParser.parseXMLString(transformedPayload);
		serializerSource.setNode(transformedDoc);

		/* One source element may produce more than one output elements. For this reason, use
		 * the XPath expression defined in each key description to retrieve all the values that
		 * the current source element contains for each key. Then fill in the 'keys' element of
		 * the rowset produced above using each different key name and value, and add the resulting
		 * 'complete' rowset to the output data sink.
		 */
		Element elTuple = (Element) xpath.evaluate("/ROWSET/INSERT/TUPLE", transformedDoc, XPathConstants.NODE);
		Element elValue = (Element) elTuple.getElementsByTagName("VALUE").item(0);
		for (Object[] keyDesc : keyDescs) {
			String keyName = (String) keyDesc[0];
			XPathExpression keyXPath = (XPathExpression) keyDesc[1];
			NodeList keyValueList = null;
			if(sourceDataElement instanceof StrDataElement){
				keyValueList = (NodeList) keyXPath.evaluate(new InputSource(new StringReader(((StrDataElement)sourceDataElement).getStringContent())), XPathConstants.NODESET);
			}else{
				keyValueList = (NodeList) keyXPath.evaluate(new InputSource(sourceDataElement.getContent()), XPathConstants.NODESET);
			}
			if (keyValueList!=null && keyValueList.getLength()>0) {
				for (int z=0; z<keyValueList.getLength(); z++) {

					String keyValue = keyValueList.item(z).getTextContent();
					log.debug("keyValue "+z+": "+ keyValue);
					if (keyValue.trim().length() > 0) {
						/* Create a KEY element, with a KEYNAME and a KEYVALUE child elements. Set
						 * the key name as the text content of the KEYNAME element, and the current
						 * value in the KEYVALUE element. */
						Element elKeyName = transformedDoc.createElement("KEYNAME");
						elKeyName.setTextContent(keyName);
						Element elKeyValue = transformedDoc.createElement("KEYVALUE");
						elKeyValue.setTextContent(keyValue);
						Element elKey = transformedDoc.createElement("KEY");
						elKey.appendChild(elKeyName);
						elKey.appendChild(elKeyValue);
						elTuple.insertBefore(elKey, elValue);
					}
				}
			}
		}
		{
			// Add Key for "ObjectID"
			Element elKeyName = transformedDoc.createElement("KEYNAME");
			elKeyName.setTextContent("ObjectID");
			Element elKeyValue = transformedDoc.createElement("KEYVALUE");
			elKeyValue.setTextContent(sourceDataElement.getAttributeValue("ContentOID"));
			Element elKey = transformedDoc.createElement("KEY");
			elKey.appendChild(elKeyName);
			elKey.appendChild(elKeyValue);
			elTuple.insertBefore(elKey, elValue);
		}
		{
			// Add Key for "gDocCollectionID"
			Element elKeyName = transformedDoc.createElement("KEYNAME");
			elKeyName.setTextContent("gDocCollectionID");
			Element elKeyValue = transformedDoc.createElement("KEYVALUE");
			elKeyValue.setTextContent(sourceDataElement.getAttributeValue("ContentCollectionID"));
			Element elKey = transformedDoc.createElement("KEY");
			elKey.appendChild(elKeyName);
			elKey.appendChild(elKeyValue);
			elTuple.insertBefore(elKey, elValue);
		}		
		{
			// Add Key for "gDocCollectionLang"
			Element elKeyName = transformedDoc.createElement("KEYNAME");
			elKeyName.setTextContent("gDocCollectionLang");
			Element elKeyValue = transformedDoc.createElement("KEYVALUE");
			elKeyValue.setTextContent(sourceDataElement.getAttributeValue("language"));
			Element elKey = transformedDoc.createElement("KEY");
			elKey.appendChild(elKeyName);
			elKey.appendChild(elKeyValue);
			elTuple.insertBefore(elKey, elValue);
		}		

		{
			// Add !!!!_Field_!!!! for "ObjectID"
			Element elfield = transformedDoc.createElement("FIELD");
			elfield.setTextContent(sourceDataElement.getAttributeValue("ContentOID"));
			elfield.setAttribute("name","ObjectID");
			elValue.appendChild(elfield);
		}
		{
			// Add !!!!_Field_!!!! for "gDocCollectionID"
			Element elfield = transformedDoc.createElement("FIELD");
			elfield.setTextContent(sourceDataElement.getAttributeValue("ContentCollectionID"));
			elfield.setAttribute("name","gDocCollectionID");
			elValue.appendChild(elfield);
		}
		{
			// Add !!!!_Field_!!!! for "gDocCollectionLang"
			Element elfield = transformedDoc.createElement("FIELD");
			elfield.setTextContent(sourceDataElement.getAttributeValue("language"));
			elfield.setAttribute("name","gDocCollectionLang");
			elValue.appendChild(elfield);
		}
		


		/* Transform the document to String format */
		StringWriter sw = new StringWriter();
		StreamResult sresult = new StreamResult(sw);
		serializer.transform(serializerSource, sresult);
		String result = sw.getBuffer().toString();

		/* Add the serialized document to the data sink */
		//		destElement = sink.getNewDataElement(sourceElement, result);
		//		if (destElement instanceof ResultSetDataElement) {
		//			((ResultSetDataElement) destElement).setDocumentID(sourceElement.getContentObjectID());
		//			((ResultSetDataElement) destElement).setCollectionID(sourceElement.getMetadataCollectionID());
		//		}
		//		sink.writeNext(destElement);

		transformedElement.setContent(result);
		return transformedElement;
	}

}
