package org.gcube.datatransformation.datatransformationlibrary.reports;

import java.util.HashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.gcube.datatransformation.datatransformationlibrary.security.DTSSManager;
import org.gcube.common.searchservice.searchlibrary.resultset.elements.ResultElementGeneric;
import org.gcube.common.searchservice.searchlibrary.rsclient.elements.RSLocator;
import org.gcube.common.searchservice.searchlibrary.rsclient.elements.RSResourceWSRFType;
import org.gcube.common.searchservice.searchlibrary.rswriter.RSXMLWriter;

/**
 * @author Dimitris Katris, NKUA
 * 
 * <p><tt>Report</tt> is the class responsible to maintain all the {@link Record}s for one transformationUnit.</p> 
 */
public class Report {

	protected HashMap<String, Record> records = new HashMap<String, Record>();
	
	private RSXMLWriter writer;
	
	private static Logger log = LoggerFactory.getLogger(Report.class);
	
	protected String reportID;
	
	/**
	 * Initializes a new <tt>Report</tt>.
	 * 
	 * @throws Exception If the {@link RSXMLWriter} in which the <tt>Report</tt> will be written could not be created.
	 */
	public Report() throws Exception {
		try {
			writer = RSXMLWriter.getRSXMLWriter();
			writer.setRecsPerPart(1);
			writer.setTimer(120000, 180000 , 10000);
		} catch (Exception e) {
			log.error("Could not create RSXMLWriter",e);
			throw new Exception("Could not create RSXMLWriter");
		}
	}
	
	/**
	 * Returns the endpoint of the <tt>ResultSet</tt> which contains the <tt>Report</tt>.
	 * 
	 * @return The endpoint of the <tt>ResultSet</tt> which contains the <tt>Report</tt>.
	 * @throws Exception If the {@link RSLocator} of the <tt>ResultSet</tt> could not be retrieved.
	 */
	public String getReportEndpoint() throws Exception {
		try {
			log.debug("Creating RSLocator at scope "+DTSSManager.getScope().toString());
			return writer.getRSLocator(new RSResourceWSRFType(), DTSSManager.getScope()).getLocator();
		} catch (Exception e) {
			log.error("Could not create RSLocator of the report result set", e);
			throw new Exception("Could not create RSLocator of the report result set");
		}
	}
	
	private Record createRecord(String objectID){
		Record record = new Record();
		record.report=this;
		record.objectID=objectID;
		records.put(objectID, record);
		return record;
	}
	
	/**
	 * Returns the <tt>Record</tt> of the given <tt>DataElement</tt> ID.
	 * 
	 * @param objectID The <tt>DataElement</tt> ID.
	 * @return The <tt>Record</tt>.
	 */
	public Record getRecord(String objectID){
		Record rec = records.get(objectID);
		if(rec==null){
			rec = createRecord(objectID);
		}
		return rec;
	}
	
	protected void commitRecord(Record record){
		//Write record...
		String payload = record.toString();
		ResultElementGeneric rselement;
		try {
			rselement = new ResultElementGeneric(record.objectID, ReportManager.getReportID(), "1.0", payload);
			writer.addResults(rselement);
			records.remove(record.objectID);
		} catch (Exception e) {
			log.error("Could not add record into result set",e);
		}
	}
	
	protected void close(){
		try {
			records=null;
			writer.close();
			writer=null;
		} catch (Exception e) {
			log.error("Could not close RSXMLWriter",e);
		}
	}
}
