package org.gcube.datatransformation.datatransformationlibrary.utils.stax;

import java.io.IOException;
import java.net.URL;
import java.util.Set;

import javax.management.modelmbean.XMLParseException;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.EndElement;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * @author jgerbe
 *
 * <p>
 *	Reader class used for reading xml with Stax 
 * </p>
 */
public class StaxReader {
	private static Logger log = LoggerFactory.getLogger(StaxReader.class);
	
	private XMLEventReader eventReader;

	private String currPath;

	private Set<String> pathsSet;

	/**
	 * Constructor class that parses xml with stax through http url
	 * 
	 * @param pathsSet Paths to match
	 * @param url http url to get content from
	 * @throws XMLStreamException in case of bad xml
	 * @throws IOException in case of bad url
	 */
	public StaxReader(Set<String> pathsSet, URL url) throws XMLStreamException, IOException {
		// First, create a new XMLInputFactory
		XMLInputFactory inputFactory = XMLInputFactory.newInstance();

		// read input stream from url and create an xml event reader
		eventReader = inputFactory.createXMLEventReader(url.openStream());

		currPath = "/";

		this.pathsSet = pathsSet;
	}

	/**
	 * Check if reader has next
	 * 
	 * @return true if reader still open
	 */
	public boolean hasNext() {
		return eventReader.hasNext();
	}

	/**
	 * get next element
	 * 
	 * @return next element
	 */
	public StaxResponse next() {
		try {

			// Read collection description
			XMLEvent event;
			while (eventReader.hasNext()) {
				event = eventReader.nextEvent();

				if (event.isStartElement()) {
					StartElement startElement = event.asStartElement();

					currPath += (currPath.endsWith("/") ? "" : "/") + startElement.getName().getLocalPart();

					if (pathsSet.contains(currPath)) {
						event = eventReader.nextEvent();

						if (!event.isCharacters()) {
							log.warn("event: " + event + " is not characters");
							continue;
						}
						StaxResponse resp = new StaxResponse(event.asCharacters().getData(), currPath);
						return resp;
					}
				}

				if (event.isEndElement()) {
					EndElement endElement = event.asEndElement();

					if (!currPath.endsWith(endElement.getName().getLocalPart()))
						throw new XMLParseException("unkows closing element tag: " + endElement.getName().getLocalPart());

					currPath = currPath.substring(0, currPath.lastIndexOf("/"));

					if (currPath.isEmpty())
						currPath = "/";
				}
			}
		} catch (XMLStreamException e) {
			log.warn("will return null", e);
			return null;
		} catch (XMLParseException e) {
			log.warn("will return null", e);
			return null;
		}
		return null;
	}
}
