package org.gcube.datatransformation.datatransformationlibrary.datahandlers.impl;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;

import org.apache.commons.io.IOUtils;
import org.gcube.common.clients.fw.queries.StatefulQuery;
import org.gcube.common.core.scope.GCUBEScopeManager;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.data.streams.Stream;
import org.gcube.data.tml.proxies.TReader;
import org.gcube.data.tml.proxies.TServiceFactory;
import org.gcube.data.trees.data.Tree;
import org.gcube.data.trees.io.XMLBindings;
import org.gcube.data.trees.patterns.Patterns;
import org.gcube.datatransformation.datatransformationlibrary.DTSCore;
import org.gcube.datatransformation.datatransformationlibrary.dataelements.DataElement;
import org.gcube.datatransformation.datatransformationlibrary.dataelements.impl.StrDataElement;
import org.gcube.datatransformation.datatransformationlibrary.datahandlers.ContentTypeDataSource;
import org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataBridge;
import org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataHandlerDefinitions;
import org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataSource;
import org.gcube.datatransformation.datatransformationlibrary.datahandlers.impl.utils.RSDataElementUtil;
import org.gcube.datatransformation.datatransformationlibrary.model.ContentType;
import org.gcube.datatransformation.datatransformationlibrary.model.Parameter;
import org.gcube.datatransformation.datatransformationlibrary.reports.Record.Status;
import org.gcube.datatransformation.datatransformationlibrary.reports.Record.Type;
import org.gcube.datatransformation.datatransformationlibrary.reports.ReportManager;
import org.gcube.datatransformation.datatransformationlibrary.security.DTSSManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This {@link DataSource} fetches {@link Tree}s from Tree Manager.
 * 
 * @author john.gerbesiotis - DI NKUA
 * 
 */
public class TMDataSource implements DataSource, ContentTypeDataSource {

	private String treeCollectionID;

	private DataBridge bridge = DTSCore.getDataBridge();

	private static Logger log = LoggerFactory.getLogger(TMDataSource.class.toString());

	private Stream<Tree> treesReader;

	/**
	 * This constructor for {@link TMDataSource}
	 * 
	 * @param input
	 *            The input value of the {@link DataSource}.
	 * @param inputParameters
	 *            The output parameters of the <tt>DataSource</tt>.
	 */
	public TMDataSource(String input, Parameter[] inputParameters) {
		treeCollectionID = input;
		log.debug("Going to fetch objects from tree manager with id: " + treeCollectionID);

		if (GCUBEScopeManager.DEFAULT.getScope() == null) {
			GCUBEScopeManager.DEFAULT.setScope(DTSSManager.getScope());
		}

		StatefulQuery query = TServiceFactory.readSource().withId(treeCollectionID).build();
		TReader treader = TServiceFactory.reader().matching(query).build();

		treesReader = treader.get(Patterns.tree());

		new Thread() {
			public void run() {
				this.setName("TMDataSourceRetriever");

				File tempIDsStorage = null;
				try {
					String seperator = UUID.randomUUID().toString();
					tempIDsStorage = File.createTempFile("DTS", ".tmp");
					log.info("File storing Trees: " + tempIDsStorage.getName());
					BufferedWriter out = new BufferedWriter(new FileWriter(tempIDsStorage));
					int i = 0;
					while (treesReader.hasNext()) {
						Tree t = treesReader.next();
						i++;
						if ((i % 100) == 0)
							log.debug("Pre-fetched " + i + " trees.");
						out.write(t.uri() + "\n");
						out.write(XMLBindings.toString(t) + "\n");
						out.write(seperator + "\n");
					}
					out.close();
					log.info("Done prefetching trees...");

					String id, str, payload = "";
					BufferedReader in = new BufferedReader(new FileReader(tempIDsStorage));
					while ((id = in.readLine()) != null) {
						if (id.isEmpty())
							break;

						str = in.readLine();
						while (!str.equals(seperator)) {
							payload += str;
							str = in.readLine();
						}

						manageObject(id, payload);
						payload = "";
					}
					in.close();

					log.info("Removing temp file.");
					tempIDsStorage.delete();
				} catch (Exception e) {
					log.error("Did not manage to fetch content from cms", e);
				} finally {
					bridge.close();
				}
			}
		}.start();

	}

	/**
	 * @see org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataHandler#close()
	 */
	@Override
	public void close() {
		bridge.close();
	}

	/**
	 * @see org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataHandler#isClosed()
	 * @return true if the {@link TReader} has been closed. Otherwise false.
	 */
	@Override
	public boolean isClosed() {
		return bridge.isClosed();
	}

	/**
	 * @see org.gcube.datatransformation.datatransformationlibrary.datahandlers.ContentTypeDataSource#nextContentType()
	 * @return next DataElement content type. application.xml in any case.
	 */
	@Override
	public ContentType nextContentType() {
		DataElement de = bridge.next();

		return de == null ? null : de.getContentType();
	}

	/**
	 * @see org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataSource#hasNext()
	 * @return true only if TReader has more trees to read. Otherwise false.
	 */
	@Override
	public boolean hasNext() {
		return bridge.hasNext();
	}

	/**
	 * @see org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataSource#next()
	 * @return next DataElement
	 */
	@Override
	public DataElement next() {
		return bridge.next();
	}

	private void manageObject(String id, String payload) {
		try {
			StrDataElement object = StrDataElement.getSourceDataElement();
			object.setId(id);
			object.setContent(payload);

			object.setAttribute(DataHandlerDefinitions.ATTR_COLLECTION_ID, treeCollectionID);
			object.setAttribute(DataHandlerDefinitions.ATTR_CONTENT_OID, id);

			object.setContentType(new ContentType("application/xml", new ArrayList<Parameter>()));

			bridge.append(object);
			ReportManager.manageRecord(id, "Object with id " + id + " was added for processing by TM", Status.SUCCESSFUL, Type.SOURCE);
		} catch (Exception e) {
			log.error("Could not manage to fetch the object " + id, e);
			ReportManager.manageRecord(id, "Object with id " + id + " could not be fetched TM", Status.FAILED, Type.SOURCE);
		}
	}

//	public static void main(String[] args) throws Exception {
//		ScopeProvider.instance.set("/gcube/devNext");
//		DTSSManager.setScope("/gcube/devNext");
//		String input = "7f740c92-13a6-45b1-95b3-4beb58085e4f";
//		DataSource ds = new TMDataSource(input, null);
//		
//		 while(ds.hasNext()){
//			 StringWriter writer = new StringWriter();
//			 IOUtils.copy(ds.next().getContent(), writer, "UTF-8");
//			 String theString = writer.toString();
//			 
//			 System.out.println(theString);
//
//			 Thread.sleep(60000);
//}
////		FtsRowset_Transformer transformer = new FtsRowset_Transformer();
//
//
//		List<DataSource> sources = new ArrayList<DataSource>();
//		sources.add(ds);
//
//		while (ds.hasNext())
//			System.out.println(RSDataElementUtil.stringFromInputStream(ds.next().getContent()));
//
//		Thread.sleep(3600 * 1000);
//		List<Parameter> programParameters = new ArrayList<Parameter>();
//		// programParameters.add(new Parameter("xslt",
//		// "bb099010-f2c8-11dd-99ef-cbe8b682b1c1"));
//		// programParameters.add(new Parameter("finalftsxslt",
//		// "821167b0-8b78-11e0-a9c6-9c00829f1447"));
//		programParameters.add(new Parameter("xslt:1", "$BrokerXSLT_DwC-A_anylanguage_to_ftRowset_anylanguage"));
//		programParameters.add(new Parameter("xslt:2", "$BrokerXSLT_TAXONOMY_anylanguage_to_ftRowset_anylanguage"));
//		programParameters.add(new Parameter("xslt:3", "$BrokerXSLT_PROVENANCE_anylanguage_to_ftRowset_anylanguage"));
//		programParameters.add(new Parameter("finalftsxslt", "$BrokerXSLT_wrapperFT"));
//		programParameters.add(new Parameter("indexType", "haha_2.0"));
//
//		ContentType targetContentType = new ContentType();
//		targetContentType.setMimeType("text/xml");
//		targetContentType.setContentTypeParameters(Arrays.asList(new Parameter[] { new Parameter("schemaURI", "http://ftrowset.xsd") }));
//
//		PathDataSink sink = new PathDataSink("/home/jgerbe/testArea/sink", null);
////		transformer.transform(sources, programParameters, targetContentType, sink);
//		sink.getOutput();
//		Thread.sleep(60 * 60 * 1000);
//	}
}