package org.gcube.datatransformation.datatransformationlibrary.datahandlers.impl;

import gr.uoa.di.madgik.grs.buffer.IBuffer.Status;
import gr.uoa.di.madgik.grs.proxy.local.LocalWriterProxy;
import gr.uoa.di.madgik.grs.proxy.tcp.TCPWriterProxy;
import gr.uoa.di.madgik.grs.record.GenericRecord;
import gr.uoa.di.madgik.grs.record.GenericRecordDefinition;
import gr.uoa.di.madgik.grs.record.RecordDefinition;
import gr.uoa.di.madgik.grs.record.field.Field;
import gr.uoa.di.madgik.grs.record.field.FieldDefinition;
import gr.uoa.di.madgik.grs.record.field.FileField;
import gr.uoa.di.madgik.grs.record.field.FileFieldDefinition;
import gr.uoa.di.madgik.grs.record.field.StringField;
import gr.uoa.di.madgik.grs.record.field.StringFieldDefinition;
import gr.uoa.di.madgik.grs.utils.Locators;
import gr.uoa.di.madgik.grs.writer.GRS2WriterException;
import gr.uoa.di.madgik.grs.writer.RecordWriter;

import java.io.File;
import java.net.URI;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.gcube.datatransformation.datatransformationlibrary.dataelements.DataElement;
import org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataSink;
import org.gcube.datatransformation.datatransformationlibrary.datahandlers.impl.utils.RSDataElementUtil;
import org.gcube.datatransformation.datatransformationlibrary.model.Parameter;
import org.gcube.datatransformation.datatransformationlibrary.reports.ReportManager;

/**
 * This {@link DataSink} stores {@link DataElement}s in a result set.
 * 
 * @author john.gerbesiotis - DI NKUA
 * 
 */
public class GRS2DataSink implements DataSink {

	private static Logger log = LoggerFactory.getLogger(GRS2DataSink.class);

	private RecordDefinition[] defs = null;
	private Boolean flush = false;
	private int wroterecs = 0;
	private boolean deleteOnDisposep = false;
	private RecordWriter<GenericRecord> writer = null;

	/**
	 * This constructor is used when DTS is instantiating a new gRS2 Data Sink.
	 * 
	 * @param output
	 *            The output value of the <tt>DataSink</tt>.
	 * @param outputParameters
	 *            The output parameters of the <tt>DataSink</tt>.
	 * @throws Exception
	 *             If the result set could not be created.
	 */
	public GRS2DataSink(String output, Parameter[] outputParameters) throws Exception {
		if(outputParameters!=null){
			for(Parameter param: outputParameters){
				if(param!=null && param.getName()!=null && param.getValue()!=null){
					if(param.getName().equalsIgnoreCase("deleteOnDispose")){
						try {
							deleteOnDisposep = Boolean.parseBoolean(param.getValue());
							log.debug("GRS2DataSink will be set with deleteOnDispose set to " + deleteOnDisposep);
						} catch (Exception e) { }
					}

				}
			}
		}

		
		FileFieldDefinition ffd = new FileFieldDefinition("File");
		ffd.setDeleteOnDispose(deleteOnDisposep);
		defs = new RecordDefinition[] { new GenericRecordDefinition((new FieldDefinition[] { new StringFieldDefinition("Metadata"), ffd })) };

		writer = new RecordWriter<GenericRecord>(new TCPWriterProxy(), defs, RecordWriter.DefaultBufferCapacity,
				RecordWriter.DefaultConcurrentPartialCapacity, RecordWriter.DefaultMirrorBufferFactor, 1, TimeUnit.DAYS);
	}

	/**
	 * @see org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataSink#append(org.gcube.datatransformation.datatransformationlibrary.dataelements.DataElement)
	 */
	public void append(DataElement dataElement) {
		/* Wrap the element's payload in a new ResultElementGeneric */
		GenericRecord rec = new GenericRecord();
		try {

			File f = RSDataElementUtil.dataElementContentToFile(dataElement);
			if (f != null)
				if (!f.exists()){
					log.warn(f.getAbsolutePath() + " (No such file or directory)");
					return;
				}
			rec.setFields(new Field[] { new StringField(RSDataElementUtil.dataElementMetadataToXML(dataElement)), new FileField(f) });

			int hours = 1;
			while (!writer.put(rec, 1, TimeUnit.HOURS)) {
				Status s = writer.getStatus();
				hours++;
				log.debug("RS buffer full.... looping... " + dataElement.getId() + " status " + s.toString());
				if (hours > 25) {
					log.info("Transformation has remained idle for one day. Closing writer.");
					try {
						writer.close();
					} catch (GRS2WriterException e) {
						break;
					}
					writer = null;
					flush = true;
				}
			}

			wroterecs++;
			log.debug("Wrote record (" + wroterecs + ") #" + dataElement.getId());
		} catch (Exception e) {
			if (flush)
				log.info("Flushing output.");
			else
				log.error("Failed to create RS2 element", e);

			return;
			// throw new Exception("Failed to create ResultSet element.", e);
		}

	}

	/**
	 * @see org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataHandler#close()
	 */
	public void close() {
		try {
			isClosed = true;
			log.debug("Total records added: " + writer.totalRecords());
			writer.close();
			ReportManager.closeReport();
		} catch (Exception e) {
			log.error("Could not close GRS2DataSink", e);
		}

	}

	/**
	 * @see org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataSink#getOutput()
	 * @return The output of the transformation.
	 */
	public String getOutput() {
		try {
			return writer.getLocator().toASCIIString();
		} catch (Exception e) {
			log.error("Did not manage to create the RS Locator", e);
			return null;
		}
	}

	private boolean isClosed = false;

	/**
	 * @see org.gcube.datatransformation.datatransformationlibrary.datahandlers.DataHandler#isClosed()
	 * @return true if the <tt>DataHandler</tt> has been closed.
	 */
	public boolean isClosed() {
		return isClosed;
	}

}
