package org.gcube.gcat.persistence.ckan;

import static org.gcube.resources.discovery.icclient.ICFactory.clientFor;
import static org.gcube.resources.discovery.icclient.ICFactory.queryFor;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.ws.rs.InternalServerErrorException;
import javax.ws.rs.WebApplicationException;

import org.gcube.common.encryption.encrypter.StringEncrypter;
import org.gcube.common.resources.gcore.ServiceEndpoint;
import org.gcube.common.resources.gcore.ServiceEndpoint.AccessPoint;
import org.gcube.common.resources.gcore.ServiceEndpoint.Property;
import org.gcube.gcat.utils.ContextUtility;
import org.gcube.resources.discovery.client.api.DiscoveryClient;
import org.gcube.resources.discovery.client.queries.api.SimpleQuery;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class CKANInstance {
	
	private static final Logger logger = LoggerFactory.getLogger(CKANInstance.class);
	
	// CKAN Instance info
	private final static String RUNTIME_CATALOGUE_RESOURCE_NAME = "CKanDataCatalogue";
	private final static String PLATFORM_CATALOGUE_NAME = "Tomcat";
	
	// property to retrieve the master service endpoint into the /root scope
	private final static String IS_MASTER_ROOT_KEY_PROPERTY = "IS_ROOT_MASTER"; // true, false.. missing means false as well
	
	private final static String API_KEY_PROPERTY = "API_KEY";
	private final static String SOCIAL_POST = "SOCIAL_POST";
	private final static String ALERT_USERS_ON_POST_CREATION = "ALERT_USERS_ON_POST_CREATION";
	private final static String URL_RESOLVER = "URL_RESOLVER";
	
	private static final Map<String,CKANInstance> ckanInstancePerScope;
	
	protected String ckanURL;
	protected String sysAdminToken;
	protected boolean socialPostEnabled;
	protected boolean notificationToUsersEnabled;
	protected String uriResolverURL;
	
	static {
		ckanInstancePerScope = new HashMap<String,CKANInstance>();
	}
	
	public static CKANInstance getInstance() {
		CKANInstance ckanInstance = ckanInstancePerScope.get(ContextUtility.getCurrentContext());
		if(ckanInstance == null) {
			ckanInstance = new CKANInstance();
			ckanInstance.getConfigurationFromIS();
			ckanInstancePerScope.put(ContextUtility.getCurrentContext(), ckanInstance);
		}
		return ckanInstance;
	}
	
	/**
	 * Retrieve endpoints information from IS for DataCatalogue URL
	 * @return list of endpoints for ckan data catalogue
	 * @throws Exception
	 */
	private static List<ServiceEndpoint> getServiceEndpoints() {
		SimpleQuery query = queryFor(ServiceEndpoint.class);
		query.addCondition("$resource/Profile/Name/text() eq '" + RUNTIME_CATALOGUE_RESOURCE_NAME + "'");
		query.addCondition("$resource/Profile/Platform/Name/text() eq '" + PLATFORM_CATALOGUE_NAME + "'");
		DiscoveryClient<ServiceEndpoint> client = clientFor(ServiceEndpoint.class);
		List<ServiceEndpoint> serviceEndpoints = client.submit(query);
		if(serviceEndpoints.size() == 0) {
			logger.error("There is no {} having name {} and Platform {} in this context.",
					ServiceEndpoint.class.getSimpleName(), RUNTIME_CATALOGUE_RESOURCE_NAME, PLATFORM_CATALOGUE_NAME);
			throw new InternalServerErrorException("No CKAN configuration on IS");
		}
		return serviceEndpoints;
	}
	
	private void getConfigurationFromIS() {
		try {
			List<ServiceEndpoint> serviceEndpoints = getServiceEndpoints();
			ServiceEndpoint serviceEndpoint = null;
			
			if(serviceEndpoints.size() > 1) {
				logger.info("Too many {} having name {} in this context. Looking for the one that has the property {}",
						ServiceEndpoint.class.getSimpleName(), RUNTIME_CATALOGUE_RESOURCE_NAME,
						IS_MASTER_ROOT_KEY_PROPERTY);
				
				for(ServiceEndpoint se : serviceEndpoints) {
					Iterator<AccessPoint> accessPointIterator = se.profile().accessPoints().iterator();
					while(accessPointIterator.hasNext()) {
						ServiceEndpoint.AccessPoint accessPoint = accessPointIterator.next();
						
						// get the is master property
						Property entry = accessPoint.propertyMap().get(IS_MASTER_ROOT_KEY_PROPERTY);
						String isMaster = entry != null ? entry.value() : null;
						
						if(isMaster == null || !isMaster.equals("true")) {
							continue;
						}
						
						// set this variable
						serviceEndpoint = se;
						break;
					}
				}
				
				// if none of them was master, throw an exception
				if(serviceEndpoint == null) {
					throw new InternalServerErrorException(
							"Too many CKAN configuration on IS and no one with MASTER property");
				}
				
			} else {
				serviceEndpoint = serviceEndpoints.get(0);
			}
			
			Iterator<AccessPoint> accessPointIterator = serviceEndpoint.profile().accessPoints().iterator();
			while(accessPointIterator.hasNext()) {
				AccessPoint accessPoint = accessPointIterator.next();
				
				// add this host
				ckanURL = accessPoint.address();
				
				// retrieve sys admin token
				sysAdminToken = accessPoint.propertyMap().get(API_KEY_PROPERTY).value();
				sysAdminToken = StringEncrypter.getEncrypter().decrypt(sysAdminToken);
				
				// retrieve option to check if the social post has to be made
				socialPostEnabled = true; // default is true
				if(accessPoint.propertyMap().containsKey(SOCIAL_POST)) {
					if(accessPoint.propertyMap().get(SOCIAL_POST).value().trim().equalsIgnoreCase("false")) {
						socialPostEnabled = false;
					}
				}
				
				// retrieve option for user alert
				notificationToUsersEnabled = false;
				if(accessPoint.propertyMap().containsKey(ALERT_USERS_ON_POST_CREATION)) {
					if(accessPoint.propertyMap().get(ALERT_USERS_ON_POST_CREATION).value().trim()
							.equalsIgnoreCase("true")) {
						notificationToUsersEnabled = true;
					}
				}
				
				// retrieve URL_RESOLVER
				if(accessPoint.propertyMap().containsKey(URL_RESOLVER)) {
					uriResolverURL = accessPoint.propertyMap().get(URL_RESOLVER).value();
				}
				
			}
			
		} catch(WebApplicationException e) {
			throw e;
		} catch(Exception e) {
			throw new InternalServerErrorException("Error while getting configuration on IS", e);
		}
		
	}
	
	public String getUriResolverURL() throws Exception {
		return uriResolverURL;
	}
	
	public String getCKANURL() {
		return ckanURL;
	}
	
	public boolean isSocialPostEnabled() throws Exception {
		return socialPostEnabled;
	}
	
	public boolean isNotificationToUsersEnabled() throws Exception {
		return notificationToUsersEnabled;
	}
	
	public String getSysAdminToken() throws Exception {
		return sysAdminToken;
	}
	
}
