package org.gcube.smartgears.extensions;

import static org.gcube.common.events.impl.Utils.*;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.gcube.smartgears.extensions.HttpExtension.Method;

/**
 * Represents the complete signature for an API endpoint.
 * This class aggregates method signatures for different HTTP methods on the same URL mapping.
 * 
 * @author Luca Frosini (ISTI-CNR)
 */
public class ApiSignature {
	
	
	private final String mapping;
	private Set<Method> methods = new HashSet<Method>();
	private Map<Method,Set<String>> requestTypes = new HashMap<Method, Set<String>>();
	private Map<Method,Set<String>> responseTypes = new HashMap<Method, Set<String>>();

	
	/**
	 * Creates a new API signature for the specified URL mapping.
	 * 
	 * @param mapping the URL mapping for this API
	 */
	public ApiSignature(String mapping) {
		this.mapping=mapping;
	}
	
	/**
	 * Adds a method signature to this API signature.
	 * 
	 * @param signature the method signature to add
	 * @return this signature for method chaining
	 */
	public ApiSignature with(ApiMethodSignature signature) {
		
		notNull("method signature",signature);
		
		this.methods.add(signature.method);
		this.requestTypes.put(signature.method,signature.requestTypes);
		this.responseTypes.put(signature.method,signature.responseTypes);

		return this;
	}
	
	/**
	 * Returns the URL mapping for this API.
	 * 
	 * @return the URL mapping
	 */
	public String mapping() {
		return mapping;
	}
	
	/**
	 * Returns the set of HTTP methods supported by this API.
	 * 
	 * @return the supported HTTP methods
	 */
	public Set<Method> methods() {
		return methods;
	}
	
	/**
	 * Returns the request types accepted for each HTTP method.
	 * 
	 * @return a map of HTTP methods to their accepted request types
	 */
	public Map<Method,Set<String>> requestTypes() {
		return requestTypes;
	}
	
	/**
	 * Returns the response types produced for each HTTP method.
	 * 
	 * @return a map of HTTP methods to their produced response types
	 */
	public Map<Method,Set<String>> responseTypes() {
		return responseTypes;
	}
}