package org.gcube.smartgears.configuration.application;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import org.gcube.com.fasterxml.jackson.annotation.JsonAutoDetect;
import org.gcube.com.fasterxml.jackson.annotation.JsonAutoDetect.Visibility;
import org.gcube.com.fasterxml.jackson.annotation.JsonIgnore;
import org.gcube.com.fasterxml.jackson.annotation.JsonInclude;
import org.gcube.com.fasterxml.jackson.annotation.JsonInclude.Include;
import org.gcube.com.fasterxml.jackson.annotation.JsonProperty;
import org.gcube.common.validator.ValidationError;
import org.gcube.common.validator.Validator;
import org.gcube.common.validator.ValidatorFactory;
import org.gcube.common.validator.annotations.NotEmpty;
import org.gcube.common.validator.annotations.NotNull;
import org.gcube.smartgears.configuration.PersistenceConfiguration;

/**
 * The configuration of a managed app.
 * <p>
 * Includes the list of its client services.
 *  
 * @author Lucio Lelii
 * @author Luca Frosini (ISTI-CNR)
 *
 */
@JsonInclude(value = Include.NON_NULL)
@JsonAutoDetect(fieldVisibility = Visibility.ANY)
public class ApplicationConfiguration {

	@NotNull
	String name;

	@NotNull
	String group;

	@NotNull
	String version;
	
	String description="";
	
	@JsonIgnore
	String context;
	
	private boolean proxable = true;
		
	Set<GCubeExclude> excludes= new HashSet<>();
	
	Set<GCubeInclude> includes= new HashSet<>();
	
	@NotEmpty @JsonProperty("persistence")
	PersistenceConfiguration persistenceConfiguration; 
	
	@JsonProperty("allowed-secrets") 
	List<String> allowedSecretClasses = null;
	
	/**
	 * Returns the set of excluded packages or classes.
	 * 
	 * @return the set of exclusions
	 */
	public Set<GCubeExclude> excludes() {
		return excludes;
	}
	
	/**
	 * Returns the set of included packages or classes.
	 * 
	 * @return the set of inclusions
	 */
	public Set<GCubeInclude> includes() {
		return includes;
	}
	
	/**
	 * Default constructor.
	 */
	public ApplicationConfiguration() {}
			
	/**
	 * Returns the application name.
	 * 
	 * @return the application name
	 */
	public String name() {
		return name;
	}
	
	/**
	 * Returns the application context.
	 * 
	 * @return the application context
	 */
	public String context() {
		return context;
	}
	
	/**
	 * Sets the excluded packages or classes.
	 * 
	 * @param excludes the exclusions to set
	 * @return this configuration for method chaining
	 */
	public ApplicationConfiguration excludes(GCubeExclude ... excludes) {
		this.excludes=new HashSet<GCubeExclude>(Arrays.asList(excludes));
		return this;
	}
	
	/**
	 * Sets the included packages or classes.
	 * 
	 * @param includes the inclusions to set
	 * @return this configuration for method chaining
	 */
	public ApplicationConfiguration includes(GCubeInclude... includes) {
		this.includes=new HashSet<GCubeInclude>(Arrays.asList(includes));
		return this;
	}
	
	/**
	 * Sets the allowed secret classes.
	 * 
	 * @param classNames the class names of allowed secrets
	 * @return this configuration for method chaining
	 */
	public ApplicationConfiguration allowedSecrets(String ... classNames) {
		this.allowedSecretClasses = Arrays.asList(classNames);
		return this;
	}
	
	/**
	 * Returns the list of allowed secret classes.
	 * 
	 * @return the list of allowed secret class names
	 */
	public List<String> allowedSecrets(){
		return this.allowedSecretClasses;
	}
	
	/**
	 * Sets the application context.
	 * 
	 * @param context the context to set
	 * @return this configuration for method chaining
	 */
	public ApplicationConfiguration context(String context) {
		this.context = context;
		return this;
	}
	
	/**
	 * Sets the application name.
	 * 
	 * @param name the name to set
	 * @return this configuration for method chaining
	 */
	public ApplicationConfiguration name(String name) {
		this.name=name;
		return this;
	}
	
	/**
	 * Sets the persistence configuration.
	 * 
	 * @param configuration the persistence configuration to set
	 * @return this configuration for method chaining
	 */
	public ApplicationConfiguration persistenceConfiguration(PersistenceConfiguration configuration) {
		this.persistenceConfiguration = configuration;
		return this;
	}

	/**
	 * Sets whether the application can be proxied.
	 * 
	 * @param proxable true if the application can be proxied, false otherwise
	 * @return this configuration for method chaining
	 */
	public ApplicationConfiguration proxable(boolean proxable) {
		this.proxable = proxable;
		return this;
	}
	
	/**
	 * Returns the application group.
	 * 
	 * @return the application group
	 */
	public String group() {
		return group;
	}
	
	
	/**
	 * Sets the application group.
	 * 
	 * @param group the group to set
	 * @return this configuration for method chaining
	 */
	public ApplicationConfiguration group(String group) {
		this.group=group;
		return this;
	}
	
	/**
	 * Returns the application version.
	 * 
	 * @return the application version
	 */
	public String version() {
		return version;
	}
	
	/**
	 * Sets the application version.
	 * 
	 * @param version the version to set
	 * @return this configuration for method chaining
	 */
	public ApplicationConfiguration version(String version) {
		this.version=version;
		return this;
	}

	
	/**
	 * Returns the application description.
	 * 
	 * @return the application description
	 */
	public String description() {
		return description;
	}
	
	/**
	 * Sets the application description.
	 * 
	 * @param description the description to set
	 * @return this configuration for method chaining
	 */
	public ApplicationConfiguration description(String description) {
		this.description=description;
		return this;
	}

	/**
	 * Returns whether the application can be proxied.
	 * 
	 * @return true if the application can be proxied, false otherwise
	 */
	public boolean proxable() {
		return proxable;
	}
		
	/**
	 * Returns the persistence configuration.
	 * 
	 * @return the persistence configuration
	 */
	public PersistenceConfiguration persistenceConfiguration() {
		return persistenceConfiguration;
	}
		
	/**
	 * Validates this configuration.
	 */
	public void validate() {
		
		List<String> msgs = new ArrayList<String>();
		
		Validator validator = ValidatorFactory.validator();
		
		for (ValidationError error : validator.validate(this))
			msgs.add(error.toString());
		
		if (!this.excludes().isEmpty() && !this.includes().isEmpty())
			msgs.add("exclude tags and includes tags are mutually exclusive");
		
		if (!msgs.isEmpty())
			throw new IllegalStateException("invalid configuration: "+msgs);
	}

	@Override
	public int hashCode() {
		return Objects.hash(description, excludes, group, includes, name, proxable, version, allowedSecretClasses);
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		ApplicationConfiguration other = (ApplicationConfiguration) obj;
		return Objects.equals(description, other.description)
				&& Objects.equals(excludes, other.excludes) && Objects.equals(group, other.group)
				&& Objects.equals(includes, other.includes) && Objects.equals(name, other.name)
				&& proxable == other.proxable && Objects.equals(version, other.version);
	}

	@Override
	public String toString() {
		return "ApplicationConfiguration [name=" + name + ", group=" + group + ", version=" + version + ", description="
				+ description + ", context=" + context + ", proxable=" + proxable + ", excludes=" + excludes
				+ ", includes=" + includes + ", persistenceConfiguration=" + persistenceConfiguration
				+ ", allowedSecretClasses=" + this.allowedSecretClasses + "]";
	}
	
	
}