package org.gcube.smartgears.managers;

import static org.gcube.smartgears.lifecycle.container.ContainerState.*;
import static org.gcube.smartgears.provider.ProviderFactory.*;

import java.util.List;

import org.gcube.common.events.Observes;
import org.gcube.common.events.Observes.Kind;
import org.gcube.smartgears.configuration.container.ContainerHandlers;
import org.gcube.smartgears.context.application.ApplicationContext;
import org.gcube.smartgears.context.container.ContainerContext;
import org.gcube.smartgears.handlers.container.ContainerHandler;
import org.gcube.smartgears.handlers.container.ContainerLifecycleEvent;
import org.gcube.smartgears.handlers.container.ContainerPipeline;
import org.gcube.smartgears.lifecycle.application.ApplicationLifecycle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Coordinates management of the container as a gCube resource.
 * 
 * @author Fabio Simeoni
 * 
 */
public class ContainerManager {

	private static Logger log = LoggerFactory.getLogger(ContainerManager.class);
	
	public static ContainerManager instance = new ContainerManager();
	
	private ContainerContext context;
	
	private ContainerPipeline pipeline;
	
	private ContainerManager() {}
	/**
	 * Starts container management.
	 * 
	 */
	public ContainerContext start() {
		
		log.info("starting container management");
		
		context = null;
		
		try {
		
			context = provider().containerContext();
			
			context.configuration().validate();
			
			ContainerHandlers handlers = provider().containerHandlers();
			
			log.trace("managing container lifecycle with {}", handlers.get());
			
			startHandlers(handlers.get());
			
			context.lifecycle().moveTo(active);
			
			return context;
		}
		catch(RuntimeException e) {
			
			log.error("cannot manage container (see cause)",e);
			
			if (context!=null)
				context.lifecycle().moveTo(failed);
			
			throw e;
		}
		
	}
	
	public void manage(ApplicationContext app) {
		
		app.events().subscribe(this); 
		
	}
	
	@Observes(value={ApplicationLifecycle.failure,ApplicationLifecycle.stop},kind=Kind.critical)
	void monitorApplication(ApplicationLifecycle lifecycle) {
		
		context.lifecycle().tryMoveTo(partActive);
	}
	
	/**
	 * Stops container management.
	 * 
	 */
	public void stop() {

		if (context == null)
			return;

		log.info("stopping container management");

		try {
			
			context.lifecycle().tryMoveTo(stopped);
		
			stopHandlers();
			
			log.info("stopping container  events");
			context.events().stop();
			
			

		} 
		catch (RuntimeException e) {

			log.warn("cannot stop container management (see cause)", e);
		}

	}

	//helpers
	
	private void startHandlers(List<ContainerHandler> handlers) {

		try {

			pipeline = new ContainerPipeline(handlers);
			
			pipeline.forward(new ContainerLifecycleEvent.Start(context));

		} catch (RuntimeException e) {
			
			context.lifecycle().tryMoveTo(failed);
			throw e;
		}
	}
	
	
	private void stopHandlers() {

		if (pipeline == null)
			return;

		// copy pipeline, flip it, and
		ContainerPipeline returnPipeline = pipeline.reverse();

		// start lifetime pipeline in inverse order with stop event
		returnPipeline.forward(new ContainerLifecycleEvent.Stop(context));

	}
	
	
	

}
