package org.gcube.smartgears;

import static org.gcube.smartgears.utils.Utils.*;

import java.util.Set;

import javax.servlet.ServletContainerInitializer;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;

import org.gcube.smartgears.context.application.ApplicationContext;
import org.gcube.smartgears.context.container.ContainerContext;
import org.gcube.smartgears.managers.ApplicationManager;
import org.gcube.smartgears.managers.ContainerManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Bootstraps management of all deployed applications which require it.
 * 
 * @author Fabio Simeoni
 * 
 */
public class Bootstrap implements ServletContainerInitializer {


	private static Logger log = LoggerFactory.getLogger(Bootstrap.class);
	
	private boolean containerHasFailed = false;
	private ContainerManager manager;
	private ContainerContext context;
	
	@Override
	public void onStartup(Set<Class<?>> c, ServletContext application) throws ServletException {

		ApplicationManager appManager = new ApplicationManager();
		
		// pass through for non managed services
		if (isResource(application)) {
		
			try {
		
				initialiseContainerIfStillNeeded(application);
					
				log.info("starting management of application @ {}", application.getContextPath());
				
				ApplicationContext app = appManager.start(context,application);
				
				manager.manage(app);
				
			}
			catch(Throwable t) {
				
				appManager.stop();

				throw new ServletException("cannot manage application @ " + application.getContextPath() + " (see cause)", t);
				
			}
		}
	}
	
	
	//helpers
	
	
	private void initialiseContainerIfStillNeeded(ServletContext application) {
		
		boolean containerIsNotInitialised = context==null && !containerHasFailed;
		
		ClassLoader contextCL = Thread.currentThread().getContextClassLoader();
		
		if (containerIsNotInitialised) {
			
			try {
				
				//we initialise the container in the same classloader as this lib, lest container bind its resources to the current webapp
				Thread.currentThread().setContextClassLoader(ContainerManager.class.getClassLoader());
				
				
				manager = ContainerManager.instance;
				
				context = manager.start();
			}
			catch(RuntimeException e) {
				
				containerHasFailed=true;
				
				throw new IllegalStateException("container is not managed",e);
			}
			finally {
				//we then restore the classloader of the current application
				Thread.currentThread().setContextClassLoader(contextCL);
			}
		}
		else 
			if (containerHasFailed)
				throw new IllegalStateException("container is not managed");

		
	}
}
