package org.gcube.common.resources;

import java.util.Set;

import javax.xml.namespace.QName;

import org.w3c.dom.Element;

/**
 * A model of a managed resource .
 * 
 * @author Fabio Simeoni
 *
 */
public interface Resource {

	/**
	 * Returns the identifier of the resource
	 * @return the identifier
	 */
	String id();

	/**
	 * Returns the edit version of the resource.
	 * @return the version
	 */
	long version();

	/**
	 * Returns the name of the resource.
	 * @return the name
	 */
	QName name();

	/**
	 * Sets the name of the resource.
	 * @param name the name
	 */
	void setName(QName name);

	/**
	 * Returns the description of the resource.
	 * @return the description
	 */
	String description();

	/**
	 * Sets the description of the resource.
	 * @param description the description
	 */
	void setDescription(String description);

	/**
	 * Returns the facets of the resource.
	 * @return the facets
	 */
	Set<Facet> facets();

	/**
	 * Returns <code>true</code> if the resource has a facet with a given name.
	 * @param name the name
	 * @return <code>true</code> if the resource has a facet with the given name, <code>false</code> otherwise
	 */
	boolean hasFacet(QName name);

	/**
	 * Returns <code>true</code> if the resource has a facet bound to a given type.
	 * @param type the type
	 * @return <code>true</code> if the resource has a facet bound to the given type, <code>false</code> otherwise
	 */
	<T extends TypedFacet> boolean hasFacet(Class<T> type);

	/**
	 * Returns a facet with a given name.
	 * @param name the name
	 * @return the facet
	 * @throws IllegalArgumentException if the resource does not have a facet with the given name
	 */
	Facet facet(QName name) throws IllegalArgumentException;

	/**
	 * Returns a facet bound to a given type.
	 * @param type the type
	 * @return the facet
	 * @throws IllegalArgumentException if the resource does not have a facet bound to the given type
	 */
	<T extends TypedFacet> T facet(Class<T> type) throws IllegalArgumentException;

	/**
	 * Removes a facet with a given name.
	 * @param name the name
	 * @throws IllegalArgumentException if the resource does not have a facet with the given name
	 */
	void removeFacet(QName name) throws IllegalArgumentException;

	/**
	 * Adds a facet to the resource.
	 * @param facet the facet
	 * @throws IllegalArgumentException if the resource has already a facet with the same name
	 */
	void setFacet(Facet facet) throws IllegalArgumentException;

	/**
	 * Adds a {@link UntypedFacet} with a given data to the resource
	 * @param data the facet's data
	 * @throws IllegalArgumentException if the resource has already a facet with the same name
	 */
	public void setFacet(Element data) throws IllegalArgumentException;

}