package org.gcube.common.resources;

import static org.gcube.common.resources.Constants.*;

import java.rmi.UnmarshalException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Null;
import javax.validation.constraints.Pattern;
import javax.xml.bind.Unmarshaller;
import javax.xml.bind.annotation.XmlAnyElement;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.namespace.QName;

import org.gcube.common.resources.validation.Published;
import org.gcube.common.resources.validation.Unpublished;
import org.w3c.dom.Element;

/**
 * Default {@link Resource} implementation.
 * 
 * @author Fabio Simeoni
 *
 */
@XmlRootElement(name="resource")
public final class DefaultResource implements Resource {

	@Null(message=UNEXPECTED_ID,groups=Unpublished.class)
	@NotNull(message=NO_ID,groups=Published.class)
	@Pattern(message=BAD_ID,regexp=BAD_CHARS,groups=Published.class)
	@XmlAttribute(required=false)
	private String id;
	
	@Null(message=UNEXPECTED_VERSION,groups=Unpublished.class)
	@NotNull(message=NO_VERSION,groups=Published.class)
	@XmlAttribute(required=false)
	private Long version;
	
	@XmlElement(required=false)
	private QName name;
	
	@XmlElement(required=false)
	private String description;
	
	@XmlAnyElement(lax=true)
	@Valid 
	List<Facet> facets = new ArrayList<Facet>();
	
	DefaultResource() {}
	
	@Override
	public String id() {
		return id;
	}
	
	public void setId(String id) {
		this.id=id;
	}
	
	@Override
	public long version() {
		return version;
	}
	
	public void bumpVersion() {
		version = (version==null)?1:version+1;
	}
	
	@Override
	public QName name() {
		return name;
	}
	
	@Override
	public void setName(QName name) {
		this.name = name;
	}
	
	@Override
	public String description() {
		return description;
	}
	
	@Override
	public void setDescription(String description) {
		this.description = description;
	}
	
	@Override
	public Set<Facet> facets() {
		return new LinkedHashSet<Facet>(facets);
	}
	
	@Override
	public boolean hasFacet(QName name) {
		try {
			facet(name);
			return true;
		}
		catch(IllegalArgumentException e) {
			return false;
		}
	}
	
	@Override
	public <T extends TypedFacet> boolean hasFacet(Class<T> type) {
		try {
			facet(type);
			return true;
		}
		catch(IllegalArgumentException e) {
			return false;
		}
	}
	
	@Override
	public Facet facet(QName name) throws IllegalArgumentException {
		for (Facet facet : facets)
			if (facet.name().equals(name))
				return facet;
		
		throw new IllegalArgumentException("unknown facet "+name);
	}
	
	@Override
	public <T extends TypedFacet> T facet(Class<T> type) throws IllegalArgumentException {
		for (Facet facet : this.facets)
			if (type.isInstance(facet))
				return type.cast(facet);
		throw new IllegalArgumentException("no facets of type "+type);
	}
	
	@Override
	public void removeFacet(QName name) throws IllegalArgumentException {
		Facet facet = facet(name);
		facets.remove(facet);
	}
	
	@Override
	public void setFacet(Facet facet) {
		
		try {
			Facet existingFacet = facet(facet.name());
			facets.set(facets.indexOf(existingFacet), facet);
		}
		catch(IllegalArgumentException e) { //new facet
			this.facets.add(facet);	
		}
	}

	@Override
	public void setFacet(Element facet) {
		QName name = new QName(facet.getNamespaceURI(),facet.getLocalName());
		setFacet(new UntypedFacet(name,facet));
	}
	
	/**
	 * Invoked after resource unmarshalling, validates generic constraints on the resource
	 * @param u the unmarshaller
	 * @param parent the parent
	 */
	public void afterUnmarshal(Unmarshaller u, Object parent) throws UnmarshalException {
		Set<QName> names = new HashSet<QName>();	
	       for (Facet facet : facets)
	    	   if (names.contains(facet.name())) {
	    		  throw new UnmarshalException("resource has multiple facets with name "+facet.name());  
	    	   }
	    	   else 
	    		   names.add(facet.name());
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((description == null) ? 0 : description.hashCode());
		result = prime * result + ((facets == null) ? 0 : facets.hashCode());
		result = prime * result + ((id == null) ? 0 : id.hashCode());
		result = prime * result + ((name == null) ? 0 : name.hashCode());
		result = prime * result + ((version == null) ? 0 : version.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		DefaultResource other = (DefaultResource) obj;
		if (description == null) {
			if (other.description != null)
				return false;
		} else if (!description.equals(other.description))
			return false;
		if (facets == null) {
			if (other.facets != null)
				return false;
		} else if (!facets.equals(other.facets))
			return false;
		if (id == null) {
			if (other.id != null)
				return false;
		} else if (!id.equals(other.id))
			return false;
		if (name == null) {
			if (other.name != null)
				return false;
		} else if (!name.equals(other.name))
			return false;
		if (version == null) {
			if (other.version != null)
				return false;
		} else if (!version.equals(other.version))
			return false;
		return true;
	}

	

}
