package org.gcube.common.ghn.service.handlers;

import java.util.logging.Handler;

import org.gcube.common.ghn.service.context.ApplicationContext;
import org.gcube.common.ghn.service.events.ApplicationEvent;
import org.gcube.common.ghn.service.events.RequestEvent;
import org.gcube.common.ghn.service.events.ResponseEvent;

/**
 * A {@link Handler} of {@link RequestEvent}s and {@link ResponseEvent}s.
 * <p>
 * The handler participates in a {@link Pipeline} of other handlers registered for notification of the same events.
 * After processing the event, it may or may not propagate the event to the handlers further down in the
 * {@link Pipeline} {@link Pipeline#forward(ApplicationEvent)}.
 * 
 * @author Fabio Simeoni
 * 
 * @see RequestEvent
 * @see ResponseEvent
 * @see Pipeline
 */
public abstract class RequestHandler extends AbstractHandler<RequestHandler> {

	/**
	 * Initialises the handler.
	 * 
	 * @param ctx the servlet context of the managed app.
	 */
	public void start(ApplicationContext ctx) {
	}

	/**
	 * Invoked when the container receives a request for a servlet of a managed app.
	 * 
	 * @param pipeline the pipeline in which this handler is registered
	 * @param e the event
	 */
	public void handleRequest(RequestEvent e) {
	}

	/**
	 * Invoked when a servlet of a managed app has produced a response to a request.
	 * 
	 * @param pipeline the pipeline in which this handler is registered
	 * @param e the event
	 */
	public void handleResponse(ResponseEvent e) {
	}

	@Override
	public void onEvent(ApplicationEvent<RequestHandler> e) {

		// mind the order here, ResponseEvent<RequestEvent so must be checked first
		if (e instanceof ResponseEvent)
			handleResponse(ResponseEvent.class.cast(e));
		else if (e instanceof RequestEvent)
			handleRequest(RequestEvent.class.cast(e));

	}

	/**
	 * Terminates the handler.
	 */
	public void stop() {
	}

}
