package org.gcube.common.ghn.service.handlers;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.gcube.common.ghn.service.events.ApplicationEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * An ordered list {@link ApplicationHandler}s that handle related {@link ApplicationEvent}s.
 * 
 * @author Fabio Simeoni
 * 
 * @param <H> the type of the handlers
 * 
 * 
 */
public class Pipeline<H extends ApplicationHandler<H>> {

	private static Logger log = LoggerFactory.getLogger(Pipeline.class);

	private final List<H> handlers;
	private int cursor = 0;

	/**
	 * Creates an instance with a given list of handlers.
	 * <p>
	 * Modification to the input list do not affect the pipeline.
	 * 
	 * @param handlers the handlers
	 */
	public Pipeline(List<H> handlers) {
		this.handlers = new ArrayList<H>(handlers); // defensive copy
	}

	/**
	 * Returns the handlers in this pipeline.
	 * <p>
	 * Modification to the input list do not affect the pipeline.
	 * 
	 * @return the handlers
	 */
	public List<H> handlers() {
		return new ArrayList<H>(handlers); // defensive copy
	}

	/**
	 * Forwards an event through the pipeline.
	 * 
	 * @param event the event
	 */
	public void forward(ApplicationEvent<H> event) {

		if (cursor >= handlers.size())
			return;

		H handler = handlers.get(cursor);

		cursor++;

		log.trace("forwarding {} to {}", event, handler);

		handler.onEvent(event);

		forward(event);

	}

	/**
	 * Returns a pipeline with the same handlers as this pipeline, only in reverse order.
	 * 
	 * @return the reversed pipeline
	 */
	public Pipeline<H> reverse() {
		
		List<H> handlers = handlers(); // it's a copy, we're not changing this pipeline
		
		Collections.reverse(handlers);
		
		return new Pipeline<H>(handlers);
	}
}
