package org.gcube.common.ghn.service.configuration;

import static org.gcube.common.ghn.service.utils.Utils.*;

import java.io.InputStream;
import java.lang.reflect.Modifier;
import java.util.HashSet;
import java.util.ServiceLoader;
import java.util.Set;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;

import org.gcube.common.ghn.service.handlers.ApplicationHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Binds {@link Configuration}s to and from XML serialisations.
 * 
 * @author Fabio Simeoni
 * 
 */
public class ConfigurationBinder {

	private static final Logger log = LoggerFactory.getLogger(ConfigurationBinder.class);

	/**
	 * Returns a {@link Configuration} from its XML serialisation.
	 * 
	 * @param stream the serialisation
	 * @return the configuration
	 * @throws RuntimeException if the serialisation is invalid
	 */
	public Configuration bind(InputStream stream) {

		try {

			JAXBContext ctx = JAXBContext.newInstance(Configuration.class);

			return (Configuration) ctx.createUnmarshaller().unmarshal(stream);

		} catch (JAXBException e) {

			throw new RuntimeException("invalid service configuration", e);

		}
	}

	/**
	 * Returns a {@link Configuration} from its XML serialisation.
	 * 
	 * @param stream the serialisation
	 * @return the configuration
	 * @throws RuntimeException if the serialisation is invalid
	 */
	public Handlers bindHandlers(InputStream stream) {

		//collects  handler classes
		Set<Class<?>> classes = scanForConfigurationElements();

		try {

			JAXBContext ctx = JAXBContext.newInstance(classes.toArray(new Class<?>[0]));

			return (Handlers) ctx.createUnmarshaller().unmarshal(stream);

		} catch (JAXBException e) {

			throw unchecked(e);

		}
	}
	
	
	
	private Set<Class<?>> scanForConfigurationElements() throws RuntimeException {

		@SuppressWarnings("all")
		ServiceLoader<ApplicationHandler> handlerLoader = (ServiceLoader) ServiceLoader.load(ApplicationHandler.class);

		Set<Class<?>> scanned = new HashSet<Class<?>>();

		for (ApplicationHandler<?> handler : handlerLoader) {
			Class<?> handlerClass = handler.getClass();
			if (handlerClass.isInterface() || handlerClass.getModifiers() == Modifier.ABSTRACT)
				continue;
			else
				scanned.add(handlerClass);
		}

		log.trace("managing application with the following handlers: {}", scanned);

		//add top-level configuration
		scanned.add(Handlers.class);

		return scanned;
	}
}
