package org.gcube.common.ghn.service.configuration;

import static org.gcube.common.ghn.service.utils.Utils.*;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.validation.ConstraintViolation;
import javax.validation.Valid;
import javax.validation.Validation;
import javax.validation.Validator;
import javax.validation.ValidatorFactory;
import javax.validation.constraints.NotNull;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElementRef;
import javax.xml.bind.annotation.XmlRootElement;

import org.gcube.common.ghn.service.persistence.LocalPersistence;
import org.gcube.common.ghn.service.persistence.Persistence;

/**
 * The configuration of a managed app.
 * <p>
 * Includes the list of its client services.
 *  
 * @author Fabio Simeoni
 *
 */
@XmlRootElement(name="application")
public class Configuration {

	 private static ValidatorFactory factory = Validation.buildDefaultValidatorFactory();

	@XmlAttribute
	private Mode mode = Mode.online;
	
	@XmlElement(name="name" , required=true)
	@NotNull
	String name;

	@XmlElement(name="group", required=true)
	@NotNull
	String group;

	@XmlElement(name="version", required=true)
	@NotNull
	String version;

	@XmlElement(name="description")
	String description="";
	
	@XmlElement(name="scope")
	List<String> scopes = new ArrayList<String>();
	
	@XmlElementRef(type=LocalPersistence.class)
	@NotNull @Valid
	private Persistence persistenceManager;
	
	
	public Configuration() {}
	
	/**
	 * Returns the management {@link Mode}.
	 * @return the management mode
	 */
	public Mode mode() {
		return mode;
	}
	

	
	/**
	 * Returns the name of the service.
	 * @return the name of the service
	 */
	public String name() {
		return name;
	}
	
	/**
	 * Sets the name of the service.
	 * @param name the name of the service
	 * @return these coordinates
	 */
	public Configuration name(String name) {
		this.name=name;
		return this;
	}
	
	/**
	 * Returns the class of the service
	 * @return the class of the service
	 */
	public String serviceClass() {
		return group;
	}

	/**
	 * Sets the class of the service.
	 * @param group the class of the service
	 * @return these coordinates
	 */
	public Configuration group(String group) {
		this.group=group;
		return this;
	}
	
	/**
	 * Returns the version of the service
	 * @return the version of the service
	 */
	public String version() {
		return version;
	}
	
	/**
	 * Sets the version of the service.
	 * @param version the version of the service
	 * @return these coordinates
	 */
	public Configuration version(String version) {
		this.version=version;
		return this;
	}

	
	/**
	 * Returns the description of the service.
	 * @return the description of the service
	 */
	public String description() {
		return description;
	}
	
	/**
	 * Sets the description of the service.
	 * @param description the description of the service
	 * @return these coordinates
	 */
	public Configuration description(String description) {
		this.description=description;
		return this;
	}
	
	/**
	 * Returns the start scopes of the service
	 * @return the start scopes
	 */
	public List<String> startScopes() {
		return scopes;
	}

	/**
	 * Sets the start scopes of the service.
	 * @param scopes the start scopes of the service
	 * @return this deployment configuration
	 */
	public Configuration startScopes(String ... scopes) {
		
		notNull("start scopes",scopes);
		
		if (scopes!=null && scopes.length>0)
			this.scopes = Arrays.asList(scopes);
		
		return this;
	}
	
	/**
	 * Returns the {@link LocalPersistence} for the service.
	 * @return the manager
	 */
	public Persistence persistence() {
		return persistenceManager;
	}
	
	/**
	 * Sets the {@link LocalPersistence} for the service.
	 * @param manager the manager
	 * @return this configuration
	 */
	public Configuration persistence(Persistence manager) {
		this.persistenceManager=manager;
		return this;
	}
	
	/**
	 * Sets the management mode.
	 * @param the management mode
	 * @return this configuration
	 */
	public Configuration mode(Mode mode) {
		this.mode=mode;
		return this;
	}
	
	public void validate() {
		
		List<String> msgs = new ArrayList<String>();
		
		Validator validator = factory.getValidator();
		
		for (ConstraintViolation<Configuration> error : validator.validate(this))
			msgs.add(error.getPropertyPath()+" "+error.getMessage());
		
		if (!msgs.isEmpty())
			throw new IllegalStateException("invalid configuration: "+msgs);
		
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((description == null) ? 0 : description.hashCode());
		result = prime * result + ((group == null) ? 0 : group.hashCode());
		result = prime * result + ((mode == null) ? 0 : mode.hashCode());
		result = prime * result + ((persistenceManager == null) ? 0 : persistenceManager.hashCode());
		result = prime * result + ((scopes == null) ? 0 : scopes.hashCode());
		result = prime * result + ((name == null) ? 0 : name.hashCode());
		result = prime * result + ((version == null) ? 0 : version.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Configuration other = (Configuration) obj;
		if (description == null) {
			if (other.description != null)
				return false;
		} else if (!description.equals(other.description))
			return false;
		if (group == null) {
			if (other.group != null)
				return false;
		} else if (!group.equals(other.group))
			return false;
		if (mode != other.mode)
			return false;
		if (persistenceManager == null) {
			if (other.persistenceManager != null)
				return false;
		} else if (!persistenceManager.equals(other.persistenceManager))
			return false;
		if (scopes == null) {
			if (other.scopes != null)
				return false;
		} else if (!scopes.equals(other.scopes))
			return false;
		if (name == null) {
			if (other.name != null)
				return false;
		} else if (!name.equals(other.name))
			return false;
		if (version == null) {
			if (other.version != null)
				return false;
		} else if (!version.equals(other.version))
			return false;
		return true;
	}
	
	
	
}