package org.gcube.common.ghn.service;

import static org.gcube.common.ghn.service.lifecycle.State.*;
import static org.gcube.common.ghn.service.provider.ProviderFactory.*;

import java.util.Collection;
import java.util.List;

import javax.servlet.FilterRegistration;
import javax.servlet.ServletContext;
import javax.servlet.ServletRegistration;

import org.gcube.common.ghn.service.configuration.Handlers;
import org.gcube.common.ghn.service.context.ApplicationContext;
import org.gcube.common.ghn.service.events.LifecycleEvent;
import org.gcube.common.ghn.service.extensions.ApplicationLifetime;
import org.gcube.common.ghn.service.handlers.LifecycleHandler;
import org.gcube.common.ghn.service.handlers.Pipeline;
import org.gcube.common.ghn.service.handlers.RequestHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Coordinates management functions for a given application.
 * 
 * @author Fabio Simeoni
 * 
 */
public class ApplicationManager {

	private static Logger log = LoggerFactory.getLogger(ApplicationManager.class);

	private Pipeline<LifecycleHandler> ltPipeline;
	
	private ApplicationContext context;

	/**
	 * Starts application management functions.
	 * 
	 * @param application the context of the application
	 */
	public void start(ServletContext application) {
		
		String appname = application.getContextPath();

		log.info("starting management of application @ {}", appname);

		try {

			context = provider().contextFor(application);
			
			context.configuration().validate();
			
			Handlers handlers = provider().handlersFor(context);

			start(handlers.lifecycleHandlers());

			register(handlers.requestHandlers());

			registerManagementServlet();

			context.lifecycle().moveTo(ready);

		} catch (RuntimeException e) {

			if (context!=null)
				context.lifecycle().moveTo(failed);

			throw new RuntimeException("cannot start management of application @ " + appname + " (see cause)", e);
		}

	}

	/**
	 * Stops application management functions.
	 * 
	 * @param ctx the application context
	 */
	public void stop(ServletContext ctx) {

		if (context == null)
			return;

		log.info("stopping management of application @ {} (see cause)", context.name());

		try {
			
			stopLifetimeHandlers();
			
		} 
		catch (RuntimeException e) {

			context.lifecycle().tryMoveTo(stopped);

			log.warn("cannot stop management of application @ " + context.name() + " (see cause)", e);
		}

	}

	// helpers

	private void register(List<RequestHandler> rqHandlers ) {

		ServletContext app = context.application();
		
		Pipeline<RequestHandler> rqPipelineTemplate = new Pipeline<RequestHandler>(rqHandlers);

		// attach filters based on request pipeline to each servlet
		Collection<? extends ServletRegistration> servlets = app.getServletRegistrations().values();

		for (ServletRegistration servlet : servlets) {

			String name = servlet.getName();

			if (name.equals("default") || name.equals("jsp")) // skip page-resolving servlets
				continue;
			
			for (String mapping : servlet.getMappings()) {

				ApplicationFilter requestFilter = new ApplicationFilter(context,name, rqPipelineTemplate);

				FilterRegistration.Dynamic filter = app.addFilter(name + "-filter", requestFilter);

				filter.addMappingForUrlPatterns(null, false, mapping);
			}
		}
	}

	private void registerManagementServlet() {

		ServletContext ctx = context.application();
		// TODO deal with extensions properly
		ServletRegistration.Dynamic lifetime = ctx.addServlet(ctx.getServletContextName() + "-lifetime",
				ApplicationLifetime.class);
		lifetime.addMapping(ctx.getContextPath() + "/lifetime");
	}

	private void start(List<LifecycleHandler> handlers) {

		try {

			ltPipeline = new Pipeline<LifecycleHandler>(handlers);

			ltPipeline.forward(new LifecycleEvent.Start(context));

		} catch (RuntimeException e) {
			context.lifecycle().tryMoveTo(failed);
			throw e;
		}
	}

	private void stopLifetimeHandlers() {

		if (ltPipeline == null)
			return;

		// copy pipeline, flip it, and
		Pipeline<LifecycleHandler> returnPipeline = ltPipeline.reverse();

		// start lifetime pipeline in inverse order with stop event
		returnPipeline.forward(new LifecycleEvent.Stop(context));

	}

}
