package org.gcube.common.ghn.service;

import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;
import javax.servlet.annotation.WebListener;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A {@link ServletContextListener} which bootstraps gCube management of a web application.
 * <p>
 * In servlet containers that implement version 3.0 or later of the specification, the listener may be bound to the
 * application by a {@link ApplicationInitializer} packaged with the application or used as a shared library. If the
 * container ignores {@link ApplicationInitializer}s, the listener is bound by virtue of its {@link WebListener} annotation.
 * <p>
 * In servlet containers that implement earlier versions of the specification, the listener
 * <em>must<em> be configured in the <code>web.xml</code> deployment descriptor.
 * <p>
 * In all cases, the implementations handles redundant configurations to the effect that only only one listener can be bound to any given application.
 * 
 * @author Fabio Simeoni
 * 
 */
// we don't do much more here than trigger management and control redundant configuration
// we push orchestration of management functions down to ApplicationMananger

@WebListener
public class ApplicationListener implements ServletContextListener {

	private static Logger log = LoggerFactory.getLogger(ApplicationListener.class);

	private ServletContext ctx;
	private ApplicationManager manager;

	// flags initialisation to short-circuit redundantly configured instances
	private static final String init_flag = "app-initialized";

	@Override
	public void contextInitialized(ServletContextEvent sce) {

		ctx = sce.getServletContext();
		String appname = ctx.getContextPath();

		// we short-circuit in case of a redundantly configured instance
		// (as listeners are sequentially invoked, we do not need to worry about synchronisation)
		if (appIsAlreadyManaged()) {
			log.warn("redundant configuration for application @ {}: this listener remains inactive",appname);
			return;
		}
		
		manager = new ApplicationManager();

		manager.start(ctx);

		claimApplicationManagement();
	}

	@Override
	public void contextDestroyed(ServletContextEvent sce) {

		//another way of withdrawing for redundant configuration
		if (manager == null)
			return;

		ServletContext ctx = sce.getServletContext();
		
		manager.stop(ctx);
	}
	
	
	//helpers: just to improve readability
	
	private boolean appIsAlreadyManaged() {
		
		return ctx.getAttribute(init_flag) != null;
	}
	
	// mark initialisation to short-circuit redundantly configured instances
	private void claimApplicationManagement() {
		
		ctx.setAttribute(init_flag, true);
	}

}
