package org.gcube.common.ghn.service;

import java.io.IOException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.gcube.common.ghn.service.context.ApplicationContext;
import org.gcube.common.ghn.service.context.DefaultContext;
import org.gcube.common.ghn.service.events.RequestEvent;
import org.gcube.common.ghn.service.events.ResponseEvent;
import org.gcube.common.ghn.service.handlers.Pipeline;
import org.gcube.common.ghn.service.handlers.RequestHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A {@link Filter} that executes a {@link Pipeline} of {@link RequestHandler}s before and a client request is delivered
 * to a given servlet and before the response produced by the servlet is returned to the client.
 * 
 * @author Fabio Simeoni
 * 
 */
public class ApplicationFilter implements Filter {

	private static Logger log = LoggerFactory.getLogger(ApplicationFilter.class);

	private final ApplicationContext context;
	private final String servlet;
	private final Pipeline<RequestHandler> pipeline;

	/**
	 * Creates an instance with the name of the target servlet and a pipeline.
	 * 
	 * @param servlet the name of the servlet
	 * @param pipeline the pipeline
	 */
	public ApplicationFilter(ApplicationContext context, String servletName, Pipeline<RequestHandler> pipeline) {
		this.context = context;
		this.servlet = servletName;
		this.pipeline = pipeline;
	}

	@Override
	public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException,
			ServletException {

		HttpServletRequest httprequest = (HttpServletRequest) request;
		HttpServletResponse httpresponse = (HttpServletResponse) response;

		if (shouldExcludeRequest(httprequest))

			chain.doFilter(request, response);

		else {

			// create a per-request context with temporary properties
			ApplicationContext ctx = new DefaultContext(context);

			RequestEvent event = new RequestEvent(servlet, ctx, httprequest);

			log.info("managing request {} for application @ {}", event.uri(), context.name());

			try {
				pipeline.forward(event);
			}
			catch(Throwable t) {
				
				throw new ServletException("cannot manage request "+event.uri()+" for application "+context.name(), t);
			}

			// dispatch to other filters for this servlet
			chain.doFilter(request, response);

			ResponseEvent responseEvent = new ResponseEvent(servlet, ctx, httprequest, httpresponse);

			pipeline.reverse().forward(responseEvent);
		}
	}

	@Override
	public void init(FilterConfig filterConfig) throws ServletException {

		// propagate filter initialisation to handler
		try {

			for (RequestHandler handler : pipeline.handlers())

				handler.start(context);

		} catch (Throwable t) {

			throw new ServletException(t);

		}
	}

	@Override
	public void destroy() {

		for (RequestHandler handler : pipeline.handlers())
			try {

				handler.stop();

			} catch (Throwable t) {

				log.error("cannot terminate handler {} for application {} ", handler, context.name());
			}

	}

	// helpers

	private boolean shouldExcludeRequest(HttpServletRequest request) {

		String query = request.getQueryString();

		if ("wsdl".equals(query))
			return true;

		return false;
	}
}
