package org.gcube.common.ghn.service.handlers;

import java.util.logging.Handler;

import javax.servlet.ServletContext;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

import org.gcube.common.ghn.service.configuration.RequestHandlerAdapter;
import org.gcube.common.ghn.service.events.RequestEvent;
import org.gcube.common.ghn.service.events.ResponseEvent;
import org.gcube.common.ghn.service.events.ServiceEvent;

/**
 * A {@link Handler} of {@link RequestEvent}s and {@link ResponseEvent}s.
 * <p>
 * The handler participates in a {@link Pipeline} of other handlers registered for notification of the same events.
 * After processing the event, it may or may not propagate the event to the handlers further down in the
 * {@link Pipeline} {@link Pipeline#forward(ServiceEvent)}.
 * 
 * @author Fabio Simeoni
 * 
 * @see RequestEvent
 * @see ResponseEvent
 * @see Pipeline
 */
@XmlJavaTypeAdapter(RequestHandlerAdapter.class)
public abstract class RequestHandler extends AbstractHandler<RequestHandler> {

	/**
	 * Initialises the handler.
	 * 
	 * @param ctx the servlet context of the managed service.
	 */
	public void start(ServletContext ctx) {
	}

	/**
	 * Invoked when the container receives a request for a servlet of a managed service.
	 * 
	 * @param pipeline the pipeline in which this handler is registered
	 * @param e the event
	 */
	public void handleRequest(Pipeline<RequestHandler> pipeline, RequestEvent e) {
		pipeline.forward(e);
	}

	/**
	 * Invoked when a servlet of a managed service has produced a response to a request.
	 * 
	 * @param pipeline the pipeline in which this handler is registered
	 * @param e the event
	 */
	public void handleResponse(Pipeline<RequestHandler> pipeline, ResponseEvent e) {
		pipeline.forward(e);
	}

	@Override
	public void onEvent(Pipeline<RequestHandler> pipeline, ServiceEvent<RequestHandler> e) {

		// mind the order here, ResponseEvent<RequestEvent so must be checked first
		if (e instanceof ResponseEvent)
			handleResponse(pipeline, ((ResponseEvent) e));
		else if (e instanceof RequestEvent)
			handleRequest(pipeline, ((RequestEvent) e));

	}

	/**
	 * Terminates the handler.
	 */
	public void stop() {
	}

}
