package org.gcube.common.ghn.service.configuration;

import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.Unmarshaller;
import javax.xml.bind.annotation.XmlAnyElement;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;

import org.gcube.common.ghn.service.handlers.LifetimeHandler;
import org.gcube.common.ghn.service.handlers.RequestHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The configuration of a managed service.
 * <p>
 * Includes the list of its client services.
 *  
 * @author Fabio Simeoni
 *
 */
@XmlRootElement(name="service")
public class ServiceConfiguration {

	private static Logger log = LoggerFactory.getLogger(ServiceConfiguration.class);
	
	
	
	@XmlElement(name="lifetime-handlers")
	LifetimeHandlers lifetimeHandlers;
	
	@XmlElement(name="request-handlers")
	RequestHandlers requestHandlers;
	
	ServiceConfiguration() {}
	
	/**
	 * Creates an instance with the {@link LifetimeHandler}s and {@link RequestHandler}s.
	 * @param lifetimeHandlers the lifetime handlers
	 * @param requestHandlers the request handlers
	 */
	public ServiceConfiguration(List<LifetimeHandler> lifetimeHandlers, List<RequestHandler> requestHandlers) {
		this.lifetimeHandlers = new LifetimeHandlers(lifetimeHandlers);
		this.requestHandlers = new RequestHandlers(requestHandlers);
	}
	
	/**
	 * Returns the {@link LifetimeHandler}s configured on the service.
	 * @return the lifetime handlers
	 */
	public List<LifetimeHandler> lifetimeHandlers() {
		return lifetimeHandlers.values;
	}
	
	/**
	 * Returns the {@link RequestHandler}s configured on the service.
	 * @return the lifetime handlers
	 */
	public List<RequestHandler> requestHandlers() {
		return requestHandlers.values;
	}
	
	//helper
	//This method is called after JAXB unmarshalling, 
   void afterUnmarshal(Unmarshaller u, Object parent) {
	   
	   List<LifetimeHandler> ltHandlers = new ArrayList<LifetimeHandler>(lifetimeHandlers.values); 
	   //discard unavailable handlers
	   for (LifetimeHandler handler: ltHandlers)
		   if (handler instanceof LifetimeHandlerAdapter.UnavailableHandler) {
			   log.error("lifetime handler {} is configured but not installed",handler);
			   lifetimeHandlers.values.remove(handler);
	   }
	   
	  List<RequestHandler> rqHandlers = new ArrayList<RequestHandler>(requestHandlers.values); 

	   for (RequestHandler handler: rqHandlers)
		   if (handler instanceof RequestHandlerAdapter.UnavailableHandler) {
			   log.error("request handler {} is configured but not installed",handler);
			   requestHandlers.values.remove(handler);
		   }
   }
   
   
   //////////////// HELPER BINDING CLASSES
 //used internally to introduce level of nesting in JAXB whilst preserving arbitrary extension
 	private static class LifetimeHandlers {
 	
 		@SuppressWarnings("all")
 		LifetimeHandlers() { //needed for deserialisation
 		}
 		
 		LifetimeHandlers(List<LifetimeHandler> handlers) {
 			this.values=handlers;
 		}
 		
 		@XmlAnyElement(lax=true)
 		List<LifetimeHandler> values;
 		
 	}
 	
 	//used internally to introduce level of nesting in JAXB whilst preserving arbitrary extension
 	private static class RequestHandlers {
 	
 		@SuppressWarnings("all")
 		RequestHandlers() { //needed for deserialisation
 		}
 		
 		RequestHandlers(List<RequestHandler> handlers) {
 			this.values=handlers;
 		}
 		
 		@XmlAnyElement(lax=true)
 		List<RequestHandler> values;
 		
 	}
   
}