package org.gcube.common.ghn.service.configuration;

import java.io.InputStream;
import java.lang.reflect.Modifier;
import java.net.URL;
import java.util.HashSet;
import java.util.Set;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;

import org.gcube.common.ghn.service.Constants;
import org.gcube.common.ghn.service.handlers.LifetimeHandler;
import org.gcube.common.ghn.service.handlers.RequestHandler;
import org.gcube.common.ghn.service.handlers.ServiceHandler;
import org.reflections.Reflections;
import org.reflections.util.ClasspathHelper;
import org.reflections.util.ConfigurationBuilder;
import org.scannotation.ClasspathUrlFinder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Binds {@link ServiceConfiguration}s from XML serialisations, using
 * {@link ServiceHandler} bindings found in classpath archives that include a
 * {@link #service_marker_file}.
 * 
 * @author Fabio Simeoni
 * 
 */
public class ConfigurationBinder {

	private static final Logger log = LoggerFactory.getLogger(ConfigurationBinder.class);
	
	/**
	 * Returns a {@link ServiceConfiguration} from its XML serialisation.
	 * 
	 * @param stream
	 *            the serialisation
	 * @return the configuration
	 * @throws RuntimeException
	 *             if the serialisation is invalid
	 */
	public ServiceConfiguration bind(InputStream stream) throws RuntimeException {

		Class<?>[] classes = scanForHandlers();

		try {
			
			JAXBContext ctx = JAXBContext.newInstance(classes);
			
			return (ServiceConfiguration) ctx.createUnmarshaller().unmarshal(stream);
			
		} catch (JAXBException e) {
			throw new RuntimeException("invalid service configuration", e);
		}
	}

	//used internally to scan the classpath
	Class<?>[] scanForHandlers() throws RuntimeException {

		// gather bound classes
		Set<Class<?>> bindings = new HashSet<Class<?>>();
		
		// find bindings in classpath
		ClassLoader loader = Thread.currentThread().getContextClassLoader();

		
		URL[] archives = ClasspathUrlFinder.findResourceBases(Constants.configuration_file, loader);
		
		if (archives.length==0)
			throw new RuntimeException("no service handler found on classpath");
		
		ConfigurationBuilder b = new ConfigurationBuilder().addUrls(archives).addUrls(ClasspathHelper.forPackage("org.gcube.common.ghn.service",loader));
		
		Reflections r = new Reflections(b);

		//note: reflections will not go recursively up if this means to go outside one of the URLs.
		//that is, it will but only for one level (!!). cannot use ServiceHandler thus.
		//I would not want to anywy, as that would include subinterfaces which would break JAXB
		//hence should be filtered afterwards.
		bindings.addAll(r.getSubTypesOf(RequestHandler.class));
		bindings.addAll(r.getSubTypesOf(LifetimeHandler.class));
		
		
		//purge bindings
		Set<Class<?>> purged = new HashSet<Class<?>>();
		for (Class<?> binding : bindings)
			if (binding.isInterface() || binding.getModifiers()==Modifier.ABSTRACT)
				continue;
			else
				purged.add(binding);

		log.debug("found the following service handlers: {}",bindings);
		
		// resource is always bound
		purged.add(ServiceConfiguration.class);
				
		return purged.toArray(new Class<?>[0]);
	}
}
