package org.gcube.common.ghn.service;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.servlet.FilterRegistration;
import javax.servlet.ServletContext;
import javax.servlet.ServletRegistration;

import org.gcube.common.ghn.service.configuration.ConfigurationBinder;
import org.gcube.common.ghn.service.configuration.ServiceConfiguration;
import org.gcube.common.ghn.service.events.StartServiceEvent;
import org.gcube.common.ghn.service.events.StopServiceEvent;
import org.gcube.common.ghn.service.extensions.ServiceLifetime;
import org.gcube.common.ghn.service.handlers.DefaultPipeline;
import org.gcube.common.ghn.service.handlers.LifetimeHandler;
import org.gcube.common.ghn.service.handlers.Pipeline;
import org.gcube.common.ghn.service.handlers.RequestHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Starts and stops management functions for a given service. 
 * 
 * @author Fabio Simeoni
 *
 */
public class ServiceManager {

	private static Logger log = LoggerFactory.getLogger(ServiceManager.class);

	private Pipeline<LifetimeHandler> ltPipeline;
	
	/**
	 * Starts service management functions.
	 * 
	 * @param ctx the context of the service
	 * @throws RuntimeException if the management cannot be started
	 */
	public void start(ServletContext ctx) throws RuntimeException {

		log.info("managing {}", ctx.getServletContextName());
		
		ServiceConfiguration configuration = null;
				
		//read service configuration
		try {
			ConfigurationBinder binder = new ConfigurationBinder();
			
			configuration = binder.bind(ctx.getResourceAsStream("/"+Constants.configuration_file));
			
		}
		catch(Throwable t) {
			throw new RuntimeException(t); 
		}
		
		//constructs lifetime pipelines from configuration
		List<LifetimeHandler> ltHandlers = configuration.lifetimeHandlers();
		ltPipeline = new DefaultPipeline<LifetimeHandler>(ltHandlers);
		
		//start lifetime pipeline with start event, and wait for completion
		ltPipeline.forward(new StartServiceEvent(ctx));
		
		//constructs lifetime pipelines from configuration
		List<RequestHandler> rqHandlers = configuration.requestHandlers();
		Pipeline<RequestHandler> rqPipeline = new DefaultPipeline<RequestHandler>(rqHandlers);
		
		//attach filters based on request pipeline to each servlet
		Collection<? extends ServletRegistration> servlets = ctx.getServletRegistrations().values();
		for (ServletRegistration servlet : servlets) {
			String name = servlet.getName();
			if (name.equals("default") || name.equals("jsp")) //skip page-resolving servlets
				continue;
			for (String mapping : servlet.getMappings()) {
				FilterRegistration.Dynamic filter = ctx.addFilter(name + "-filter", new RequestFilter(name,rqPipeline));
				filter.addMappingForUrlPatterns(null, false, mapping);
			}
		}
	
		//TODO deal with extensions properly
		ServletRegistration.Dynamic lifetime = ctx.addServlet(ctx.getServletContextName() + "-lifetime",
				ServiceLifetime.class);
		lifetime.addMapping(ctx.getContextPath() + "/lifetime");

		
		
	}
	
	/**
	 * Stops management functions.
	 * 
	 * @param ctx the service context
	 */
	public void stop(ServletContext ctx) {
		
		//copy pipeline, flip it, and 
		List<LifetimeHandler> reversedHandlers = ltPipeline.handlers();
		Collections.reverse(reversedHandlers);
		Pipeline<LifetimeHandler> returnPipeline = new DefaultPipeline<LifetimeHandler>(reversedHandlers);
		
		//start lifetime pipeline in inverse order with stop event
		returnPipeline.forward(new StopServiceEvent(ctx));
	}
}
