package org.gcube.common.ghn.service;

import java.io.IOException;
import java.util.Collections;
import java.util.List;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;

import org.gcube.common.ghn.service.events.RequestEvent;
import org.gcube.common.ghn.service.events.ResponseEvent;
import org.gcube.common.ghn.service.handlers.DefaultPipeline;
import org.gcube.common.ghn.service.handlers.Pipeline;
import org.gcube.common.ghn.service.handlers.RequestHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A {@link Filter} that executes a {@link Pipeline} of {@link RequestHandler}s when {@link RequestEvent}s and
 * {@link ResponseEvent}s are generated for a given servlet of a managed service.
 * 
 * @author Fabio Simeoni
 * 
 * @see Pipeline
 * @see RequestHandler
 * @see RequestEvent
 * @see ResponseEvent
 */
public class RequestFilter implements Filter {

	private static Logger log = LoggerFactory.getLogger(RequestFilter.class);
	
	private final String servletName;
	private final Pipeline<RequestHandler> pipeline;

	/**
	 * Creates an instance with the name of the target servlet and a pipeline.
	 * 
	 * @param servletName the name of the servlet
	 * @param pipeline the pipeline
	 */
	public RequestFilter(String servletName, Pipeline<RequestHandler> pipeline) {
		this.servletName = servletName;
		this.pipeline = pipeline;
	}

	@Override
	public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException,
			ServletException {

		try {

			// we protect the configured pipeline from mutations that may occur within this request
			// both to: 1) allow per-request modification, and 2) support multi-threaded access
			Pipeline<RequestHandler> requestPipeline = new DefaultPipeline<RequestHandler>(pipeline.handlers());

			//execute pipeline
			RequestEvent event = new RequestEvent(servletName, request);
			requestPipeline.forward(event);
			
			//dispatch to other filters for this servlet
			chain.doFilter(request, response);

			// flip request pipeline and execute it 
			ResponseEvent responseEvent = new ResponseEvent(servletName, request, response);
			Pipeline<RequestHandler> returnPipeline = flip(requestPipeline);
			returnPipeline.forward(responseEvent);

		} catch (Throwable t) {
			throw new ServletException(t);
		}

	}

	@Override
	public void init(FilterConfig filterConfig) throws ServletException {

		try {
			for (RequestHandler h : pipeline.handlers())
				h.start(filterConfig.getServletContext());
		} catch (Throwable t) {
			throw new ServletException(t);
		}
	}

	@Override
	public void destroy() {
		
		for (RequestHandler h : pipeline.handlers())
			try {
				h.stop();
			} catch (Throwable t) {
				log.error("cannot terminate handler {}",h.name());
			}

	}

	// helper
	Pipeline<RequestHandler> flip(Pipeline<RequestHandler> pipeline) {
		List<RequestHandler> reversedHandlers = pipeline.handlers();
		Collections.reverse(reversedHandlers);
		return new DefaultPipeline<RequestHandler>(reversedHandlers);
	}
}
