package org.gcube.common.ghn.service;

import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;
import javax.servlet.annotation.WebListener;

/**
 * A {@link ServletContextListener} for managed services which notifies a {@link ServiceManager} of service lifetime events.
 * <p>
 * In servlet containers that implement version 3.0 or later of the specification,
 * the listener is attached to the managed service by a {@link ServiceInitializer}
 * packaged within the service or a shared library. If the container ignores 
 * {@link ServiceInitializer}s, then the listener is attached to the managed service
 * by virtue of its {@link WebListener} annotation.
 * <p>
 * In servlet containers that implement earlier versions of the specification,
 * the listener <em>must<em> be configured in <code>web.xml</code>.
 * <p>
 * In all cases, the implementations handles redundant configurations,
 * in that all the listener instances after the first one short-circuit their behaviour.
 * 
 * @author Fabio Simeoni
 * 
 */
@WebListener
public class LifetimeListener implements ServletContextListener {

	ServiceManager manager;

	//flags initialisation to short-circuit redundantly configured instances
	private static final String init_flag = "service-initialized";
	
	@Override
	public void contextInitialized(ServletContextEvent sce) {

		ServletContext ctx = sce.getServletContext();

		//short-circuit in case of a redundantly configured instance
		if (ctx.getAttribute(init_flag)!=null)
			return;
		
		//create service manager and notifies service has started
		manager = new ServiceManager();

		try {
			
			manager.start(ctx);
			
		} catch (Throwable t) {
			throw new RuntimeException(t);
		}

		//mark initialisation to short-circuit redundantly configured instances
		ctx.setAttribute(init_flag,true);
	}

	@Override
	public void contextDestroyed(ServletContextEvent ctx) {

		//operates only if the instances was not short-circuited
		if (manager!=null)
			manager.stop(ctx.getServletContext());
		
	}

}
