package org.gcube.common.clients;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A {@link Client} that discovers the services with which it interacts.
 * <p>
 *  
 * Discovery clients relies on a {@link Query} to discover services.
 * They then interact with all the services they discover until they succeed with one.
 * Finally, they cache the endpoint of this service and use it first for subsequent interactions.
 * 
 * <p>
 * Note that discovery clients abandon their best-effort strategy when {@link Call}s or individual
 * failures are marked with {@link Unrecoverable}.
 * 
 * @author Fabio Simeoni
 *
 * @param <S> the type of the services
 * @param <E> the type of the service endpoints
 * 
 * @see Query
 * @see Unrecoverable
 */
public class DiscoveryClient<S,E> implements Client<S> {

	private static Logger logger = LoggerFactory.getLogger(DiscoveryClient.class);
	
	private ServiceProvider<S,E> provider;
	private Query<E> query;
	private EndpointCache<E> cache;
	
	/**
	 * Creates an instance with a {@link ServiceProvider}, a {@link Query}, and an {@link EndpointCache}.
	 * @param provider the provider
	 * @param query the query
	 * @param cache the cache
	 */
	public DiscoveryClient(ServiceProvider<S,E> provider, Query<E> query,EndpointCache<E> cache) {
		this.provider=provider;
		this.query=query;
		this.cache=cache;
	}
	
	@Override
	public <CR> CR make(Call<S,CR> call) throws Exception {
		
		//try with each endpoint in turn
		Exception lastFailure = null;
				
		//use cache first, if any
		E cached = cache.get(provider.name()); 
		if (cached!=null)
			try {
				logger.info("calling {} @ {} (cached)",provider.name(),cached);
				return call.call(provider.service(cached));
			}
			catch(Exception t) {
				
				cache.clear(provider.name());
				
				if (isUnrecoverable(call) || isUnrecoverable(t)) //exit now
					throw t;
				
				lastFailure=t;
			}

		//launch query
		List<E> results = query.fire();
		if (results.size()==0)
			throw new DiscoveryException("no service found");
		
		//exclude cached endpoint (we do not use 'remove' in case list implementation does not support it
		List<E> endpoints = excludeFromQuery(cached, results);
		
		if (endpoints.size()==0)
			throw lastFailure;
		
		//try with each endpoint in turn
		for (E endpoint : endpoints)
			try {
				logger.info("calling {} @ {}",provider.name(),endpoint);
				CR result = call.call(provider.service(endpoint));
				cache.put(provider.name(), endpoint);
				return result;
			}
			catch(Exception t) {
				lastFailure = t;
				
				if (isUnrecoverable(call) || isUnrecoverable(t)) //exit now
					break;
			}
		
		throw lastFailure;
	}
	
	private <CR> boolean isUnrecoverable(Exception e) {
		
		try {
			return e.getClass().isAnnotationPresent(Unrecoverable.class);
		}
		catch (Exception ex) {
			throw new RuntimeException(ex);
		}
	}

	private <CR> boolean isUnrecoverable(Call<S,CR> call) {
		
		try {
			return call.getClass().getMethod("call",Object.class).isAnnotationPresent(Unrecoverable.class);
		}
		catch (Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	//helper
	private List<E> excludeFromQuery(E cached,List<E> results) {
		List<E> endpoints = new ArrayList<E>();
		for (E result : results)
			if (!result.equals(cached))
				endpoints.add(result);
		return endpoints;
	}
}
