package org.gcube.common.clients;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Implements {@link EndpointCache}.
 * 
 * @author Fabio Simeoni
 *
 * @param <E> the type of the cached endpoints
 */
public class BaseEndpointCache<E> implements EndpointCache<E> {
	
	private static Logger logger = LoggerFactory.getLogger(BaseEndpointCache.class);
	
	
	/** Service map. */
	private Map<String,E> cache = Collections.synchronizedMap(new HashMap<String,E>());

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void clear() {
		logger.debug("clearing cache {}",this);
		cache.clear();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void clear(String name) {
		
		check(name);
		
		logger.debug("clearing cache {} for {}",this,name);
		
		cache.put(name, null);	
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public E get(String name) throws IllegalArgumentException {
		
		check(name);
		
		return cache.get(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void put(String name, E endpoint) throws IllegalArgumentException {
		
		check(name);
		
		if(endpoint==null)
			throw new IllegalArgumentException("service endpoint is null");
		
		logger.debug("caching {} for {}",endpoint,name);
		
		cache.put(name,endpoint);
		
	}

	//helper
	private void check(String name) throws IllegalArgumentException {
		if (name==null)
			throw new IllegalArgumentException("service name is null");
	}
	
}
