package org.gcube.contentmanagement.codelistmanager.managers.handlers;

import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.Hashtable;
import org.gcube.common.dbinterface.CastObject;
import org.gcube.common.dbinterface.Condition;
import org.gcube.common.dbinterface.attributes.AggregatedAttribute;
import org.gcube.common.dbinterface.attributes.AggregationFunctions;
import org.gcube.common.dbinterface.attributes.BooleanAttribute;
import org.gcube.common.dbinterface.attributes.SimpleAttribute;
import org.gcube.common.dbinterface.conditions.ListSelect;
import org.gcube.common.dbinterface.conditions.NotInOperator;
import org.gcube.common.dbinterface.conditions.ORCondition;
import org.gcube.common.dbinterface.conditions.OperatorCondition;
import org.gcube.common.dbinterface.pool.DBSession;
import org.gcube.common.dbinterface.queries.Select;
import org.gcube.common.dbinterface.queries.Update;
import org.gcube.common.dbinterface.queries.alters.ModifyColumnType;
import org.gcube.common.dbinterface.tables.SimpleTable;
import org.gcube.common.dbinterface.types.Type;
import org.gcube.common.dbinterface.utils.Utility;
import org.gcube.contentmanagement.codelistmanager.entities.TableField;
import org.gcube.contentmanagement.codelistmanager.entities.TableField.ColumnType;
import org.gcube.contentmanagement.codelistmanager.exception.ColumnTypeNotSelectableException;
import org.gcube.contentmanagement.codelistmanager.exception.NullValuesOnCastException;
import org.gcube.contentmanagement.codelistmanager.exception.ParentNotSelectedException;
import org.gcube.contentmanagement.codelistmanager.exception.ValueNotFoundException;
import org.gcube.contentmanagement.codelistmanager.util.ColumnReference;
import org.gcube.contentmanagement.codelistmanager.util.Constants;
import org.gcube.contentmanagement.codelistmanager.util.csv.ImportUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CodeListHandler implements CodeListCurationHandler{

	private static final Logger logger = LoggerFactory.getLogger(CodeListHandler.class);
	
	private static CodeListHandler codelistHandler = new CodeListHandler();
	
	public static CodeListHandler get(){
		return codelistHandler;
	}
	
	
	public boolean replaceValue(String fieldId, String newValue, int rowId, SimpleTable table,
			Hashtable<String, TableField> fieldLabelMapping)
			throws Exception {
		if (isParentField(fieldId, fieldLabelMapping)) return setParent(fieldId, newValue, rowId, table, fieldLabelMapping);
		else {
			Type type = table.getFieldsMapping().get(fieldId);
			Select select = DBSession.getImplementation(Select.class);
			CastObject cast= Utility.getCast(newValue, type);
			cast.setUseCastFunction(true);
			select.setAttributes(new BooleanAttribute("field", new OperatorCondition<CastObject, Object>(cast, null," IS NOT ") ));
			DBSession session = null;
			try{
				session = DBSession.connect();
				ResultSet rs = select.getResults(session, false);
				rs.next();
				if (!rs.getBoolean(1)) return false;
				
				int[] tempLength = fieldLabelMapping.get(fieldId).getLength();
				logger.debug(" fields Length is "+tempLength.length);
				if (newValue.length()>tempLength[0] || 
						ImportUtil.getAfterDotLength(newValue)>tempLength[1]){
					if (newValue.length()>tempLength[0]) tempLength[0] = newValue.length();
					if (ImportUtil.getAfterDotLength(newValue)>tempLength[1]) tempLength[1] = ImportUtil.getAfterDotLength(newValue);
					type.setPrecision(tempLength);			
					changeColumnDataType(fieldId, type, table, fieldLabelMapping);
				}
				
				Update update = DBSession.getImplementation(Update.class);
				update.setTable(table);
				update.setOperators(new OperatorCondition<SimpleAttribute, CastObject>(new SimpleAttribute(fieldId), cast, "="));
				update.setFilter(new OperatorCondition<SimpleAttribute, Integer>(new SimpleAttribute(Constants.ID_LABEL), rowId, "="));
				update.execute(session);
				return update.getAffectedLines()>0;
			}catch (Exception e) {
				logger.error("an error occuring trying to replace value",e);
				return false;
			}finally{
				if (session!=null)session.release();
			}
		}
	}

	/**
	 * 
	 * @param fieldId
	 * @return
	 */
	private  boolean isParentField(String fieldId, Hashtable<String, TableField> fieldLabelMapping){
		return fieldLabelMapping.get(fieldId).getColumnReference().getType()==ColumnType.ParentCode;
	}
	
	/**
	 * 
	 * @param parentValue is the new value for column parentCode (that exists in code column)
	 * @param rowId
	 * @return
	 * @throws ParentNotSelectedException
	 * @throws ValueNotFoundException
	 * @throws Exception
	 */
	private boolean setParent(String parentFieldId, String parentValue, int rowId, SimpleTable table,
			Hashtable<String, TableField> fieldLabelMapping) throws ColumnTypeNotSelectableException, ValueNotFoundException, Exception{
				
		TableField parentCode= null, code= null;
		for (TableField tf : fieldLabelMapping.values()){
			if (tf.getColumnReference().getType()== ColumnType.Code){
				code = tf;
				break;
			}
		}
		
		parentCode = fieldLabelMapping.get(parentFieldId);
		
		//CodeList codeList = CodeList.get(this.parentInfo.getCodeListRefId());		
		Select select = DBSession.getImplementation(Select.class);
		select.setTables(table);
		select.setAttributes(new AggregatedAttribute("*",AggregationFunctions.COUNT));
		CastObject castObject= Utility.getCast(parentValue, table.getFieldsMapping().get(code.getId()));
		select.setFilter(new OperatorCondition<SimpleAttribute, CastObject>(new SimpleAttribute(code.getId()),castObject, "="));
		DBSession session = null;
		try{
			session = DBSession.connect();
			ResultSet result = select.getResults(session);
			result.next();
			if (result.getInt(1)==0) throw new ValueNotFoundException();
			
			//updating the table
			Update update= DBSession.getImplementation(Update.class);
			update.setTable(table);
			update.setOperators(new OperatorCondition<SimpleAttribute, CastObject >(new SimpleAttribute(parentCode.getId()),castObject,"="));
			update.setFilter(new OperatorCondition<SimpleAttribute, Integer>(new SimpleAttribute(Constants.ID_LABEL), rowId,"="));
			update.execute(session);
			
			if (update.getAffectedLines()>0) return true;
			else return false;
		}finally{
			if (session!=null) session.release();
		}
	}
	
	/**
	 * 
	 * @param fieldId
	 * @param columnType
	 * @throws ParentNotSelectableException
	 */
	public void changeColumnType(String fieldId, ColumnType columnType, SimpleTable table,
			Hashtable<String, TableField> fieldLabelMapping, String ... referenceCL) throws ColumnTypeNotSelectableException{
		if (columnType == ColumnType.HLChildCode || columnType == ColumnType.HLParentCode) throw new ColumnTypeNotSelectableException();		
		if(columnType==ColumnType.ParentCode){
			String codeFieldId=null;
			for (TableField tf : fieldLabelMapping.values())
				if (tf.getColumnReference().getType()== ColumnType.Code){
					codeFieldId = tf.getId();
					break;
				}
			
			if (codeFieldId==null){
				logger.warn("the Code is not set ... fieldId is "+fieldId+" and the column type is "+columnType.toString());
				throw new ColumnTypeNotSelectableException();
			}
		}
		if(columnType==ColumnType.Code){
			if (this.hasParent(fieldLabelMapping) &&
				fieldLabelMapping.get(fieldId).getColumnReference().getType() == ColumnType.Code){
				String parentFieldId=null;
				for (TableField tf : fieldLabelMapping.values()){
					if (tf.getColumnReference().getType()== ColumnType.ParentCode) parentFieldId = tf.getId();
					break;
				}
				fieldLabelMapping.get(parentFieldId).setColumnReference(new ColumnReference(ColumnType.Undefined));
			}
		}
		fieldLabelMapping.get(fieldId).setColumnReference(new ColumnReference(columnType));
		try {
			table.initializeFieldMapping();
		} catch (Exception e) {
			logger.warn("cannot reinitilaize field mapping");
		}
	}

	/**
	 * 
	 * @return
	 */
	private boolean hasParent(Hashtable<String, TableField> fieldLabelMapping){
		for (TableField field :fieldLabelMapping.values())
			if (field.getColumnReference().getType()==ColumnType.ParentCode) return true;
		return false;
	}
	
	public boolean checkRelationMatch(String relationFieldId, SimpleTable table,
			Hashtable<String, TableField> fieldLabelMapping) {
		DBSession session= null;
		try{
			TableField parentCode= null, code= null;
			for (TableField tf : fieldLabelMapping.values()){
				if (tf.getColumnReference().getType()== ColumnType.Code){
					code = tf;
					break;
				}
			}
			parentCode = fieldLabelMapping.get(relationFieldId);
			
			if (parentCode==null) return true;
			
			session = DBSession.connect();
			Select select = DBSession.getImplementation(Select.class);
			select.setAttributes(new AggregatedAttribute("*", AggregationFunctions.COUNT));
						
			Select subSelect = DBSession.getImplementation(Select.class);
			subSelect.setAttributes(new SimpleAttribute(code.getId()));
			subSelect.setTables(table);
			
			CastObject cast= Utility.getCast(new SimpleAttribute(parentCode.getId()), new Type(code.getDataType()));
			cast.setUseCastFunction(true);
			
			select.setTables(table);
			select.setFilter(new NotInOperator(new SimpleAttribute(cast.getCast()),new ListSelect(subSelect)));
		
			ResultSet rs =select.getResults(session, false);
			rs.next();
			if (rs.getInt(1)>0) return false;
		}catch (Exception e) {
			
			logger.debug("error matching parents",e);
			return false;
		}finally{
			if (session!=null) session.release();
		}
		return true;
	}

	public Integer[] checkInvalidValues(String fieldId, Type type, SimpleTable table,
			Hashtable<String, TableField> fieldLabelMapping) throws Exception{
		DBSession session=null;
		try{
			session = DBSession.connect();
			Select select = DBSession.getImplementation(Select.class);
			select.setAttributes(new SimpleAttribute(Constants.ID_LABEL));
			CastObject cast= Utility.getCast(new SimpleAttribute(fieldId), type);
			cast.setUseCastFunction(true);
			
			Condition cond = new OperatorCondition<CastObject, Object>(cast, null," is ");
			
			if (fieldLabelMapping.get(fieldId).getColumnReference().getType()==ColumnType.ParentCode){
				TableField code= null;
				for (TableField tf : fieldLabelMapping.values()){
					if (tf.getColumnReference().getType()== ColumnType.Code){
						code = tf;
						break;
					}
				}
				Select opSelect= DBSession.getImplementation(Select.class);
				opSelect.setTables(table);
				opSelect.setAttributes(new SimpleAttribute(code.getId()));
				cond = new ORCondition(cond, new NotInOperator(new SimpleAttribute(cast.getCast()), new ListSelect(opSelect)));
			}
			
			select.setFilter(cond);
			select.setTables(table);
			logger.debug(select.getExpression());
			ResultSet result = select.getResults(session);
			ArrayList<Integer> toReturn= new ArrayList<Integer>();
			while (result.next())
				toReturn.add(result.getInt(1));
			return toReturn.toArray(new Integer[toReturn.size()]);
		}catch (Exception e) {
			logger.error("error evaluating the query",e);
			throw e;
		}finally{
			if (session!=null) session.release();
		}
	}
	
	public Integer[] checkInvalidValues(String fieldId, ColumnReference columnReference, SimpleTable table,
			Hashtable<String, TableField> fieldLabelMapping) throws Exception{
		DBSession session=null;
		if (columnReference.getType()!=ColumnType.ParentCode) throw new Exception();//TODO new Exception
		try{
			
			TableField code= null;
			for (TableField tf : fieldLabelMapping.values()){
				if (tf.getColumnReference().getType()== ColumnType.Code){
					code = tf;
					break;
				}
			}
			
			session = DBSession.connect();
			Select select = DBSession.getImplementation(Select.class);
			select.setAttributes(new SimpleAttribute(Constants.ID_LABEL));
			CastObject cast= Utility.getCast(new SimpleAttribute(fieldId), new Type(code.getDataType()));
			cast.setUseCastFunction(true);
			
			logger.debug("checkInvalidValues: codeType is "+code.getDataType().getValue());
			Condition cond = new OperatorCondition<CastObject, Object>(cast, null," is ");
				
			Select opSelect= DBSession.getImplementation(Select.class);
			opSelect.setTables(table);
			opSelect.setAttributes(new SimpleAttribute(Utility.getCast(new SimpleAttribute(code.getId()), new Type(code.getDataType())).getCast()));
			cond = new ORCondition(cond, new NotInOperator(new SimpleAttribute(cast.getCast()), new ListSelect(opSelect)));
			
			select.setFilter(cond);
			select.setTables(table);
			logger.debug(select.getExpression());
			ResultSet result = select.getResults(session);
			ArrayList<Integer> toReturn= new ArrayList<Integer>();
			while (result.next())
				toReturn.add(result.getInt(1));
			return toReturn.toArray(new Integer[toReturn.size()]);
		}catch (Exception e) {
			logger.error("error evaluating the query",e);
			throw e;
		}finally{
			if (session!=null) session.release();
		}
	}
	
	public boolean changeColumnDataType(String fieldId, Type type,
			SimpleTable table, Hashtable<String, TableField> fieldLabelMapping)
			throws NullValuesOnCastException {
		DBSession session=null;
		int nullValues;
		try{
			nullValues= checkInvalidValues(fieldId, type,  table, fieldLabelMapping).length;
		}catch (Exception e) {
			return false;
		}
		if (nullValues>0) throw new NullValuesOnCastException(); 
		try{
			session = DBSession.connect();
			ModifyColumnType changeColumnType= DBSession.getImplementation(ModifyColumnType.class);
			changeColumnType.setColumn(new SimpleAttribute(fieldId));
			changeColumnType.setNewType(type);
			changeColumnType.setTable(table);
			changeColumnType.setUseCast(true);
			changeColumnType.execute(session);
			fieldLabelMapping.get(fieldId).setDataType(type.getType());
			fieldLabelMapping.get(fieldId).setLength(type.getPrecisionArray());
			table.initializeFieldMapping(session);
		}catch (Exception e) {
			logger.error("error changing column type ",e);
			return false;
		}finally{
			if (session!=null) session.release();
		}
		return true;
	}


	public boolean isMappingFinished(SimpleTable table, 
			Hashtable<String, TableField> fieldLabelMapping) {
		for (TableField field :fieldLabelMapping.values()){
			if (field.getColumnReference().getType()==ColumnType.Undefined) return false;
			if ((field.getColumnReference().getType()==ColumnType.ParentCode ) 
					&& !checkRelationMatch(field.getId(), table, fieldLabelMapping)) return false ;
		}	
		
		return true;
	}

}
