package org.gcube.common.keycloak;

import java.net.URL;
import java.util.List;

import org.gcube.common.keycloak.model.TokenResponse;
import org.gcube.common.scope.api.ScopeProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public interface KeycloakClient {

    Logger logger = LoggerFactory.getLogger(KeycloakClient.class);

    String CATEGORY = "Auth";
    String NAME = "IAM";
    String DESCRIPTION = "oidc-token endpoint";

    /**
     * Finds the keycloak endpoint {@link URL} discovering it in the current scope provided by {@link ScopeProvider}
     * @return the keycloak endpoint URL in the current scope
     * @throws KeycloakClientException if something goes wrong discovering the endpoint URL
     */
    URL findTokenEndpointURL() throws KeycloakClientException;

    /**
     * Queries an UMA token from the Keycloak server, by using provided authorization, for the given audience (context),
     * in URLEncoded form or not, and optionally a list of permissions.
     * 
     * @param tokenUrl the token endpoint {@link URL} of the OIDC server
     * @param authorization the authorization to be set as header (e.g. a "Basic ...." auth or an encoded JWT access token preceded by the "Bearer " string)
     * @param audience the audience (context) where to request the issuing of the ticket (URLEncoded)
     * @param permissions a list of permissions, can be <code>null</code>
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryUMAToken(URL tokenURL, String authorization, String audience, List<String> permissions)
            throws KeycloakClientException;

    /**
     * Queries an UMA token from the Keycloak server, by using provided clientId and client secret for the given audience
     * (context), in URLEncoded form or not, and optionally a list of permissions.
     * 
     * @param tokenURL the token endpoint {@link URL} of the Keycloak server
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param audience the audience (context) where to request the issuing of the ticket
     * @param permissions a list of permissions, can be <code>null</code>
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryUMAToken(URL tokenURL, String clientId, String clientSecret, String audience,
            List<String> permissions)
            throws KeycloakClientException;

    /**
     * Queries an UMA token from the discovered Keycloak server in the current scope, by using provided clientId and client secret
     * for the given audience (context), in URLEncoded form or not, and optionally a list of permissions.
     * 
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param audience the audience (context) where to request the issuing of the ticket
     * @param permissions a list of permissions, can be <code>null</code>
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryUMAToken(String clientId, String clientSecret, String audience, List<String> permissions)
            throws KeycloakClientException;

    /**
     * Queries an UMA token from the discovered Keycloak server in the current scope, by using provided clientId and client secret
     * for the current scope audience (context), in URLEncoded form or not, and optionally a list of permissions.
     * 
     * @param clientId the client id
     * @param clientSecret the client secret
     * @param permissions a list of permissions, can be <code>null</code>
     * @return the issued token as {@link TokenResponse} object
     * @throws KeycloakClientException if something goes wrong performing the query
     */
    TokenResponse queryUMAToken(String clientId, String clientSecret, List<String> permissions)
            throws KeycloakClientException;
}