package org.gcube.common.homelibrary.jcr.workspace;

import java.io.File;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Workspace;
import javax.jcr.query.QueryManager;
import javax.jcr.query.QueryResult;

import org.apache.jackrabbit.util.ISO9075;
import org.apache.jackrabbit.util.Text;
import org.gcube.common.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.common.homelibrary.home.workspace.WorkspaceFolder;
import org.gcube.common.homelibrary.home.workspace.WorkspaceItem;
import org.gcube.common.homelibrary.home.workspace.WorkspaceItemType;
import org.gcube.common.homelibrary.home.workspace.exceptions.InsufficientPrivilegesException;
import org.gcube.common.homelibrary.home.workspace.exceptions.ItemAlreadyExistException;
import org.gcube.common.homelibrary.home.workspace.exceptions.ItemNotFoundException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WorkspaceFolderNotFoundException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WrongDestinationException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WrongItemTypeException;
import org.gcube.common.homelibrary.home.workspace.folder.FolderBulkCreator;
import org.gcube.common.homelibrary.home.workspace.folder.items.ExternalFile;
import org.gcube.common.homelibrary.home.workspace.folder.items.ExternalImage;
import org.gcube.common.homelibrary.home.workspace.folder.items.ExternalPDFFile;
import org.gcube.common.homelibrary.home.workspace.folder.items.ExternalResourceLink;
import org.gcube.common.homelibrary.home.workspace.folder.items.ExternalUrl;
import org.gcube.common.homelibrary.home.workspace.folder.items.Query;
import org.gcube.common.homelibrary.home.workspace.folder.items.QueryType;
import org.gcube.common.homelibrary.home.workspace.folder.items.Report;
import org.gcube.common.homelibrary.home.workspace.folder.items.ReportTemplate;
import org.gcube.common.homelibrary.home.workspace.folder.items.TabularDataLink;
import org.gcube.common.homelibrary.home.workspace.folder.items.TabularDataLink.Provenance;
import org.gcube.common.homelibrary.home.workspace.folder.items.WorkflowReport;
import org.gcube.common.homelibrary.home.workspace.folder.items.WorkflowTemplate;
import org.gcube.common.homelibrary.home.workspace.folder.items.gcube.Annotation;
import org.gcube.common.homelibrary.home.workspace.folder.items.gcube.Document;
import org.gcube.common.homelibrary.home.workspace.folder.items.gcube.ImageDocument;
import org.gcube.common.homelibrary.home.workspace.folder.items.gcube.Metadata;
import org.gcube.common.homelibrary.home.workspace.folder.items.gcube.PDFDocument;
import org.gcube.common.homelibrary.home.workspace.folder.items.gcube.UrlDocument;
import org.gcube.common.homelibrary.home.workspace.folder.items.gcube.link.DocumentLink;
import org.gcube.common.homelibrary.home.workspace.folder.items.gcube.link.ImageDocumentLink;
import org.gcube.common.homelibrary.home.workspace.folder.items.gcube.link.PDFDocumentLink;
import org.gcube.common.homelibrary.home.workspace.folder.items.ts.TimeSeries;
import org.gcube.common.homelibrary.jcr.repository.JCRRepository;
import org.gcube.common.homelibrary.jcr.repository.external.GCUBEStorage;


public abstract class JCRAbstractWorkspaceFolder extends JCRWorkspaceItem implements WorkspaceFolder {

	public JCRAbstractWorkspaceFolder(JCRWorkspace workspace, Node node) throws RepositoryException {
		super(workspace,node);
	}

	/* (non-Javadoc)
	 * @see org.gcube.common.homelibrary.jcr.workspace.JCRWorkspaceItem#getRemotePath()
	 */
	@Override
	public String getRemotePath() throws InternalErrorException {
		String remotePath = null;
		List<? extends WorkspaceItem> children = getChildren();
		for (WorkspaceItem child: children){
			if (!child.isFolder()){
				String path = child.getRemotePath();
				remotePath = path.substring(0, path.lastIndexOf('/')+1);
				break;
			}
		}

		return remotePath;
	}

	public JCRAbstractWorkspaceFolder(JCRWorkspace workspace, Node node,
			String name, String description) throws RepositoryException  {		
		super(workspace,node,name,description);
	}

	public JCRWorkspace getWorkspace() {
		return workspace;
	}


	@Override
	public abstract WorkspaceItemType getType();


	@Override
	public boolean exists(String name) throws InternalErrorException {

		try {
			return workspace.exists(name, getId());
		} catch (ItemNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongItemTypeException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public WorkspaceItem find(String name) throws InternalErrorException {
		try {
			return workspace.find(name, getId());
		} catch (ItemNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongItemTypeException e) {
			throw new InternalErrorException(e);
		}
	}


	@Override
	public List<WorkspaceItem> getChildren() {
		//		long time = System.nanoTime();

		List<WorkspaceItem> children = new ArrayList<WorkspaceItem>();
		Session session = null;
		try {

			session = JCRRepository.getSession();
			Node folderNode = session.getNodeByIdentifier(identifier);


			NodeIterator iterator = folderNode.getNodes();

			while(iterator.hasNext()) {

				Node node = iterator.nextNode();
				String path = node.getPath();
				String name = path.substring(path.lastIndexOf('/') + 1);

				try { 

					if(path.equals((workspace).trashPath) || (path.equals((workspace).mySpecialFoldersPath)) || (name.startsWith("rep:")) || (name.startsWith("hl:")) || (name.startsWith("."))) { 
						continue;
					}

					JCRWorkspaceItem item = (workspace).getWorkspaceItem(node);
					children.add(item);

				} catch (InternalErrorException e) {
					// TODO logger
					//						logger.error("Node not retrieved with type " + nodeType,e);
				} catch (RepositoryException e) {
					// TODO logger
					// logger.error("Node not retrieved with type " + nodeType,e);
				}
			}
			//				System.out.println("Time to create WorkspaceItems: "+(System.nanoTime()-time2)+" ns");
		} catch (RepositoryException e) {
			logger.error("No nodes retrieved",e);
		} catch (InternalErrorException e) {
			logger.error("Permission denied",e);
		}
		finally {
			if (session != null)
				session.logout();
		}
		//		System.out.println("Time getChildren: "+(System.nanoTime()-time)+" ns");
		return children;
	}

	@Override
	public WorkspaceFolder createFolder(String name, String description)
			throws InternalErrorException, InsufficientPrivilegesException,
			ItemAlreadyExistException {

		try {
			return workspace.createFolder(name, description, getId());
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		} catch (ItemNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		}	 
	}

	@Override
	public ExternalImage createExternalImageItem(String name,
			String description, String mimeType, InputStream imageData)
					throws InsufficientPrivilegesException, InternalErrorException,
					ItemAlreadyExistException {
		try {
			//			System.out.println("*****  createExternalImageItem with inputstream" );
			return workspace.createExternalImage(name, description, mimeType,
					imageData, getId());
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public ExternalImage createExternalImageItem(String name,
			String description, String mimeType, File tmpFile) throws InsufficientPrivilegesException,
			InternalErrorException, ItemAlreadyExistException {
		try {
			//			System.out.println("*****  createExternalImageItem with tmp file: " + tmpFile );
			return workspace.createExternalImage(name, description, mimeType,
					getId(), tmpFile);
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public ExternalFile createExternalFileItem(String name, String description,
			String mimeType, InputStream fileData)
					throws InsufficientPrivilegesException, InternalErrorException,
					ItemAlreadyExistException {

		try {
			return workspace.createExternalFile(name, description, mimeType, fileData,
					getId());
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public ExternalFile createExternalFileItem(String name, String description,
			String mimeType, File tmpFile)
					throws InsufficientPrivilegesException, InternalErrorException,
					ItemAlreadyExistException {

		try {
			return workspace.createExternalFile(name, description, mimeType, tmpFile,
					getId());
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public ExternalPDFFile createExternalPDFFileItem(String name,
			String description, String mimeType, InputStream fileData)
					throws InsufficientPrivilegesException, InternalErrorException,
					ItemAlreadyExistException {
		try {
			return workspace.createExternalPDFFile(name, description, mimeType, fileData,
					getId());
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public ExternalPDFFile createExternalPDFFileItem(String name,
			String description, String mimeType, File tmpFile)
					throws InsufficientPrivilegesException, InternalErrorException,
					ItemAlreadyExistException {
		try {
			return workspace.createExternalPDFFile(name, description, mimeType, tmpFile,
					getId());
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public ExternalUrl createExternalUrlItem(String name, String description,
			String url) throws InsufficientPrivilegesException,
			InternalErrorException, ItemAlreadyExistException{

		try {
			return workspace.createExternalUrl(name, description, url,
					getId());
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public ExternalUrl createExternalUrlItem(String name, String description,
			InputStream url) throws InsufficientPrivilegesException,
			InternalErrorException, ItemAlreadyExistException{

		try {
			return workspace.createExternalUrl(name, description, url,
					getId());
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public ExternalUrl createExternalUrlItem(String name, String description,
			File tmpFile) throws InsufficientPrivilegesException,
			InternalErrorException, ItemAlreadyExistException{

		try {
			return workspace.createExternalUrl(name, description, tmpFile,
					getId());
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public ReportTemplate createReportTemplateItem(String name,
			String description, Calendar created, Calendar lastEdit,
			String author, String lastEditBy, int numberOfSections,
			String status, InputStream templateData)
					throws InsufficientPrivilegesException, InternalErrorException,
					ItemAlreadyExistException {

		try {

			return workspace.createReportTemplate(name, description, created, lastEdit,
					author, lastEditBy, numberOfSections, status, templateData, getId());

		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public Report createReportItem(String name, String description,
			Calendar created, Calendar lastEdit, String author,
			String lastEditBy, String templateName, int numberOfSections,
			String status, InputStream reportData)
					throws InsufficientPrivilegesException, InternalErrorException,
					ItemAlreadyExistException {

		try {
			return workspace.createReport(name, description, created, lastEdit,
					author, lastEditBy, templateName, numberOfSections, status,
					reportData,getId());
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public Query createQueryItem(String name, String description, String query,
			QueryType queryType) throws InsufficientPrivilegesException,
			InternalErrorException, ItemAlreadyExistException {

		try {
			return workspace.createQuery(name, description, query, queryType, getId());
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public Query createQueryItem(String name, String description,
			InputStream query, QueryType queryType)
					throws InsufficientPrivilegesException, InternalErrorException,
					ItemAlreadyExistException {

		try {
			return workspace.createQuery(name, description, query, queryType, getId());
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public WorkspaceFolder createAquaMapsItem(String name, String description,
			String mapName, String mapType, String author, int numberOfSpecies,
			String boundingBox, float psoThreshold,
			int numberOfGeneratedImages, InputStream metadata,
			Map<String, InputStream> images)
					throws InsufficientPrivilegesException,
					WorkspaceFolderNotFoundException, InternalErrorException,
					ItemAlreadyExistException, WrongDestinationException {

		return workspace.createAquaMapsItem(name, description, mapName, mapType,
				author, numberOfSpecies, boundingBox, psoThreshold, numberOfGeneratedImages,
				metadata, images, getId());
	}

	@Override
	public Annotation createAnnotationItem(String name, String description,
			String oid, Map<String, String> data)
					throws InsufficientPrivilegesException, InternalErrorException,
					ItemAlreadyExistException {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public Metadata createMetadataItem(String name, String description,
			String oid, String schema, String language, String metadata,
			String collectionName) throws InsufficientPrivilegesException,
			InternalErrorException, ItemAlreadyExistException {

		try {
			return workspace.createMetadata(name, description, oid, schema, language, metadata,
					collectionName, getId());
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public Document createDocumentItem(String name, String description,
			String oid, String mimeType, InputStream documentData,
			Map<String, String> metadata, Map<String, String> annotations,
			String collectionName) throws InsufficientPrivilegesException,
			InternalErrorException, ItemAlreadyExistException {

		try {
			return workspace.createDocument(name, description, oid, mimeType,
					documentData, metadata, annotations, collectionName, getId());
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public ImageDocument createImageDocumentItem(String name,
			String description, String oid, String mimeType,
			InputStream imageData, Map<String, String> metadata,
			Map<String, String> annotations, String collectionName)
					throws InsufficientPrivilegesException, InternalErrorException,
					ItemAlreadyExistException {

		try {
			return workspace.createImageDocument(name, description, oid, mimeType,
					imageData, metadata, annotations, collectionName, getId());
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public PDFDocument createPDFDocumentItem(String name, String description,
			String oid, String mimeType, InputStream data,
			Map<String, String> metadata, Map<String, String> annotations,
			String collectionName) throws InsufficientPrivilegesException,
			InternalErrorException, ItemAlreadyExistException {
		try {
			return workspace.createPDFDocument(name, description, oid, mimeType, data,
					metadata, annotations, collectionName, getId());
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public UrlDocument createUrlDocumentItem(String name, String description,
			String oid, String mimeType, InputStream documentData,
			Map<String, String> metadata, Map<String, String> annotations,
			String collectionName) throws InsufficientPrivilegesException,
			InternalErrorException, ItemAlreadyExistException {

		try {
			return workspace.createUrlDocument(name, description, oid, mimeType,
					documentData, metadata, annotations, collectionName, getId());
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public DocumentLink createDocumentLinkItem(String name, String description,
			String oid, Map<String, String> metadata,
			Map<String, String> annotations, String collectionName,
			String mimeType) throws InsufficientPrivilegesException,
			InternalErrorException, ItemAlreadyExistException {
		try {
			return workspace.createDocumentLink(name, description, oid, metadata,
					annotations, collectionName, mimeType, getId());
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public ImageDocumentLink createImageDocumentLinkItem(String name,
			String description, String oid, Map<String, String> metadata,
			Map<String, String> annotations, String collectionName,
			String mimeType) throws InsufficientPrivilegesException,
			InternalErrorException, ItemAlreadyExistException {

		try {
			return workspace.createImageDocumentLink(name, description, oid,
					metadata, annotations, collectionName, mimeType, getId());
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public PDFDocumentLink createPDFDocumentLinkItem(String name,
			String description, String oid, Map<String, String> metadata,
			Map<String, String> annotations, String collectionName,
			String mimeType) throws InsufficientPrivilegesException,
			InternalErrorException, ItemAlreadyExistException {

		try {
			return workspace.createPDFDocumentLink(name, description, oid,
					metadata, annotations, collectionName, mimeType, getId());
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public TimeSeries createTimeSeries(String name, String description,
			String timeseriesId, String title, String creator,
			String timeseriesDescription, String timeseriesCreationDate,
			String publisher, String sourceId, String sourceName,
			String rights, long dimension, List<String> headerLabels,
			InputStream compressedCSV) throws InsufficientPrivilegesException,
			InternalErrorException, ItemAlreadyExistException {

		try {
			return workspace.createTimeSeries(name, description, timeseriesId, title,
					creator, timeseriesDescription, timeseriesCreationDate, publisher,
					sourceId, sourceName, rights, dimension, headerLabels, compressedCSV,
					getId());
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public WorkflowReport createWorkflowReport(String name, String description,
			String workflowId, String workflowStatus, String workflowData)
					throws InsufficientPrivilegesException, InternalErrorException,
					ItemAlreadyExistException {

		try {
			return workspace.createWorkflowReport(name, description, workflowId, workflowStatus,
					workflowData, getId());
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public WorkflowTemplate createWorkflowTemplate(String name,
			String description, String workflowId, String workflowStatus,
			String workflowData) throws InsufficientPrivilegesException,
			InternalErrorException, ItemAlreadyExistException {

		try {
			return workspace.createWorkflowTemplate(name, description, workflowId, workflowStatus,
					workflowData, getId());
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public FolderBulkCreator getNewFolderBulkCreator()
			throws InternalErrorException {

		try {
			return workspace.getNewFolderBulkCreator(getId());
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongItemTypeException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public void removeChild(WorkspaceItem child) throws InternalErrorException,
	InsufficientPrivilegesException {

		try {
			workspace.removeItem(child.getId());
		} catch (ItemNotFoundException e) {
			throw new InternalErrorException(e);
		}

	}


	@Override
	public ExternalResourceLink createExternalResourceLink(String name,
			String description, String mimeType, String resourceId,
			String pluginName) throws InternalErrorException, ItemAlreadyExistException,
			InsufficientPrivilegesException {
		try {
			return workspace.createExternalResourceLink(name, description, mimeType,
					resourceId, pluginName, getId());
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public TabularDataLink createTabularDataLink(String name, String description, String tableId, String template,
			Provenance provenance, String operator, String runtimeResourceName) throws InsufficientPrivilegesException,
			InternalErrorException, ItemAlreadyExistException {

		try {
			return workspace.createTabularDataLink(name, description, tableId, template,
					provenance, operator, runtimeResourceName, getId());
		} catch (WorkspaceFolderNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongDestinationException e) {
			throw new InternalErrorException(e);
		}
	}

	@Override
	public String getUniqueName(String initialName, boolean copy)
			throws InternalErrorException {

		Session session = JCRRepository.getSession();
		try {

			Node folderNode = session.getNodeByIdentifier(getId());

			Node node = null;				
			try{
				node = getNode(folderNode, initialName); 
			}catch (Exception e) {
				return initialName;
			}

			String name = null;


			if (node!=null){
				if (copy){
					int i=1;
					while (node!=null){
						if (i==1)
							name = initialName + " (copy)";
						else
							name = initialName + " (copy " + i +")";
						node = getNode(folderNode, name);	
						i++;
					}
					return name;
				}else {		
					int i=0;
					while (node!=null){
						name = initialName+"("+i+")";
						node = getNode(folderNode, name);	
						i++;
					}
					return name;
				}
			}

		}catch (Exception e) {
			logger.error("problem creating a unique name for " + initialName);
		}
		return initialName;
	}


	/*
	 * Return the node if a node with the same name already exists in such folder
	 */
	private Node getNode(Node folderNode, String initialName) throws RepositoryException {
		Node node = null;
		String path = folderNode.getPath();
		try{
			//			System.out.println("looking for " + initialName + " in folder " + path);
			node = folderNode.getNode(initialName);

		}catch (Exception e) {
			logger.info("No item found with name \"" + initialName + "\" in folder " + path);
			return null;
		}
		return node;

	}

	@Override
	public int getCount() throws InternalErrorException {

		String remotePath = null;
		try{
			remotePath = getRemotePath();
		}catch (Exception e) {
			// TODO: handle exception
		}
		if (remotePath!=null){
			//			System.out.println("get count: " + remotePath + "- " + workspace.getOwner().getPortalLogin());
			return GCUBEStorage.getFolderTotalItems(remotePath, workspace.getOwner().getPortalLogin());
		}else
			return 0;

	}

	@Override
	public long getSize() throws InternalErrorException {
		String remotePath = null;
		try{
			remotePath = getRemotePath();
		}catch (Exception e) {
		}
		if (remotePath!=null){
			//			System.out.println("getFolderTotalVolume: " + remotePath + "- " + workspace.getOwner().getPortalLogin());
			return GCUBEStorage.getFolderTotalVolume(remotePath, workspace.getOwner().getPortalLogin());
		}
		return 0;


	}


	@Override
	public List<WorkspaceItem> getLastItems(int limit)
			throws InternalErrorException {

		Session session = JCRRepository.getSession();
		NodeIterator iterator;
		List<WorkspaceItem> list = null;
		List<String> ids = null;
		try {
			QueryManager queryManager = session.getWorkspace().getQueryManager();	

			String path = getPath();

			StringBuilder xpath =  new StringBuilder("/jcr:root/Home/" + workspace.getOwner().getPortalLogin()
					+ ISO9075.encodePath(path) +
					"//element(*,nthl:workspaceItem) order by @jcr:lastModified descending");


			logger.info("Query: " + xpath.toString());
			javax.jcr.query.Query query = queryManager.createQuery(xpath.toString(), javax.jcr.query.Query.XPATH);

			QueryResult result = query.execute();
			iterator = result.getNodes();
			list = new LinkedList<WorkspaceItem>();
			ids = new ArrayList<String>();
			int i= 0;
			while ((i<limit) && (iterator.hasNext())) {
				Node node = iterator.nextNode();

				if (isValid(node)){

					String type = node.getPrimaryNodeType().getName();

					try {

						if (!(type.equals(JCRWorkspace.NT_WORKSPACE_FOLDER)) && !(type.equals(JCRWorkspace.NT_WORKSPACE_SHARED_FOLDER)) && !(type.equals(JCRWorkspace.NT_TRASH_ITEM))) {
							String id = node.getIdentifier();
							if (!ids.contains(id)){
								list.add(workspace.getItem(id));
								ids.add(id);
								i++;
							}
						} 
					} catch (Exception e) {

					}
				}
			}
		} catch (Exception e) {
			logger.error("Error getLastItems ",e);
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}

		return list;
	}


	private boolean isValid(Node node) throws RepositoryException {

		if (node.getName().startsWith("."))
			return false;
		String nodePath = node.getPath();
		//		System.out.println("-- " + nodePath);
		String sharePath = JCRRepository.PATH_SEPARATOR + JCRRepository.SHARED_FOLDER;

		if (nodePath.startsWith(sharePath)) {
			Node sharedNode = (Node) node.getAncestor(2);

			try{
				sharedNode.getNode(JCRWorkspaceSharedFolder.USERS).getProperty(portalLogin);
				return true;
			}catch (Exception e) {
				return false;
			}
		}
		return false;

	}

}
