package org.gcube.common.homelibrary.jcr.workspace;


import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import javax.jcr.ItemExistsException;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.jackrabbit.util.Text;
import org.apache.xalan.templates.AbsPathChecker;
import org.gcube.common.homelibrary.home.Home;
import org.gcube.common.homelibrary.home.HomeManager;
import org.gcube.common.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.common.homelibrary.home.workspace.WorkspaceFolder;
import org.gcube.common.homelibrary.home.workspace.WorkspaceItemAction;
import org.gcube.common.homelibrary.home.workspace.WorkspaceItemType;
import org.gcube.common.homelibrary.home.workspace.WorkspaceSharedFolder;
import org.gcube.common.homelibrary.home.workspace.accessmanager.ACLType;
import org.gcube.common.homelibrary.home.workspace.exceptions.InsufficientPrivilegesException;
import org.gcube.common.homelibrary.home.workspace.exceptions.ItemAlreadyExistException;
import org.gcube.common.homelibrary.home.workspace.exceptions.ItemNotFoundException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WorkspaceFolderNotFoundException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WrongDestinationException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WrongItemTypeException;
import org.gcube.common.homelibrary.home.workspace.folder.items.ExternalFile;
import org.gcube.common.homelibrary.home.workspace.folder.items.ExternalImage;
import org.gcube.common.homelibrary.home.workspace.folder.items.ExternalPDFFile;
import org.gcube.common.homelibrary.home.workspace.folder.items.gcube.ImageDocument;
import org.gcube.common.homelibrary.jcr.repository.JCRRepository;
import org.gcube.common.homelibrary.jcr.workspace.accessmanager.JCRAccessManager;
import org.gcube.common.homelibrary.jcr.workspace.accessmanager.JCRPrivilegesInfo;
import org.gcube.common.homelibrary.util.WorkspaceUtil;
import org.gcube.contentmanagement.blobstorage.transport.backend.RemoteBackendException;

public class JCRWorkspaceSharedFolder extends JCRAbstractWorkspaceFolder implements WorkspaceSharedFolder {

	public static final String USERS =	"hl:users";

	private String applicationName;
	private String destinationFolderId;
	private List<String> users;

	private String itemName;

	public JCRWorkspaceSharedFolder(JCRWorkspace workspace, Node node) throws RepositoryException {
		super(workspace,node);	
	}

	public JCRWorkspaceSharedFolder(JCRWorkspace workspace, Node node,
			String name, String description, String originalDestinationFolderId, List<String> users, String applicationName, String itemName) throws RepositoryException, InternalErrorException {		

		super(workspace,node,name,description);

		// Create a sharable node below Share root node.
		super.save(node);

		//		System.out.println("originalDestinationFolderId " + originalDestinationFolderId);
		//		System.out.println("destinationFolder " + applicationName);

		this.destinationFolderId = originalDestinationFolderId;
		this.applicationName = applicationName;
		this.itemName = itemName;
		this.users = users;

	}

	@Override
	public void save(Node node) throws RepositoryException {

		//		System.out.println("save");
		//		System.out.println("users " + users.toString());
		try {
			// The save method creates a clone of the sharable node
			//for the owner and all users below user roots.
			addUser(node,workspace.getOwner().getPortalLogin(),destinationFolderId);

			for (String user : users) {
				//				System.out.println("user " + user);
				HomeManager homeManager = workspace.getHome().getHomeManager();
				//GCUBEScope scopeOwner = workspace.getOwner().getScope();
				Home home = homeManager.getHome(user);

				if (applicationName==null){
					//					System.out.println("save application name == null");
					addUser(node, user, home.getWorkspace().getRoot().getId());
				}
				else		
				{
					//					System.out.println("save application name != null");
					addUser(node, user, home.getDataArea().getApplicationRoot(applicationName).getId());
					//					System.out.println("home.getDataArea().getApplicationRoot(applicationName).getId() " + home.getDataArea().getApplicationRoot(applicationName).getId());
				}
			}
		} catch (Exception e) {
			throw new RepositoryException(e);
		}
	}

	private Node getUserNode(Node node, String user) throws RepositoryException,
	InternalErrorException {

		//		System.out.println("get user of node: " + node.getPath());
		Node usersNode = node.getNode(USERS);
		String value = usersNode.getProperty(user).getString();

		String[] values = value.split(workspace.getPathSeparator());
		if (values.length < 2)
			throw new InternalErrorException("Path node corrupt");

		String parentId = values[0];
		String nodeName = values[1];
		Node parentNode = node.getSession().getNodeByIdentifier(parentId);

		return node.getSession().getNode(parentNode.getPath() + 
				workspace.getPathSeparator() + nodeName);
	}

	private Node getUserNode(Node node) throws RepositoryException,
	InternalErrorException {
		return getUserNode(node, workspace.getOwner().getPortalLogin());
	}

	private String getNodeName(Node node) throws RepositoryException,
	InternalErrorException {

		String[] names = node.getPath().split(
				workspace.getPathSeparator());
		return names[names.length - 1];
	}

	@Override
	public String getName() throws InternalErrorException {

		Session session = JCRRepository.getSession();
		try {
			Node sharedNode = session.getNodeByIdentifier(getId());
			Node userNode = getUserNode(sharedNode);
			return getNodeName(userNode);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}	
	}

	@Override
	public void internalRename(Node node, String newName) throws ItemAlreadyExistException, InternalErrorException {


		String nodeNewName = Text.escapeIllegalJcrChars(newName);
		try {

			Node userNode = getUserNode(node);

			if (workspace.exists(nodeNewName, userNode.getParent().getIdentifier())) {
				logger.error("Item with name " + nodeNewName + " exists");
				throw new ItemAlreadyExistException("Item " + nodeNewName + " already exists");
			}
			String newPath = userNode.getParent().getPath() 
					+ workspace.getPathSeparator() + nodeNewName;

			node.setProperty(LAST_MODIFIED,Calendar.getInstance());
			node.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
			node.setProperty(LAST_ACTION,WorkspaceItemAction.RENAMED.toString());
			node.getSession().save();

			String path = userNode.getPath();
			node.getSession().getWorkspace().move(path, newPath);

			Node usersNode = node.getNode(USERS);
			String value = userNode.getParent().getIdentifier() + workspace.getPathSeparator() + newName;
			usersNode.setProperty(workspace.getOwner().getPortalLogin(), value);


			node.getSession().save();

		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (ItemNotFoundException e) {
			throw new InternalErrorException(e);
		} catch (WrongItemTypeException e) {
			throw new InternalErrorException(e);
		} finally {

		}

	}

	@Override
	public void internalMove(Node destinationFolderNode) throws ItemAlreadyExistException,
	InternalErrorException, RepositoryException {

		try {

			logger.debug("Start internal move item with id " 
					+ getId() + " to destination item with id " + destinationFolderNode.getIdentifier());

			Session session = destinationFolderNode.getSession();

			Node node = session.getNodeByIdentifier(identifier);

			if (workspace.exists(node.getName(), destinationFolderNode.getIdentifier())) {
				logger.error("Item with name " + getName() + " exists");
				throw new ItemAlreadyExistException("Item " + node.getName() + " already exists");
			}
			node.setProperty(LAST_MODIFIED,Calendar.getInstance());
			node.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
			node.setProperty(LAST_ACTION,WorkspaceItemAction.MOVED.toString());
			node.getSession().save();

			Node userNode = getUserNode(node);
			String userNodeName = getNodeName(userNode);

			String newPath = destinationFolderNode.getPath() 
					+ workspace.getPathSeparator() + userNodeName;

			String value = destinationFolderNode.getIdentifier() +
					workspace.getPathSeparator() + userNodeName;

			session.getWorkspace().clone(session.getWorkspace().getName(),
					node.getPath(), newPath, false);


			Node usersNode = node.getNode(USERS);
			usersNode.setProperty(workspace.getOwner().getPortalLogin(), value);

			session.removeItem(userNode.getPath());

			session.save();
		} catch (ItemExistsException e) {
			throw new ItemAlreadyExistException(e.getMessage());
		} catch (RepositoryException e) {
			logger.error("Repository exception thrown by move operation",e);
			throw new InternalErrorException(e);
		} catch (WrongItemTypeException e) {
			logger.error("Unhandled Exception ");
			throw new InternalErrorException(e);
		} catch (ItemNotFoundException e) {
			logger.error("Unhandled Exception ");
			throw new InternalErrorException(e);
		} 
	}


	public Node unShareNode(Node sharedNode, Node destinationNode) throws RepositoryException,
	InternalErrorException, InsufficientPrivilegesException, ItemAlreadyExistException,
	WorkspaceFolderNotFoundException, WrongDestinationException, ItemNotFoundException,
	RemoteBackendException {

		logger.debug("unShare Node: "+ sharedNode.getPath() + " -  by user: " + workspace.getOwner().getPortalLogin());

		Session session = sharedNode.getSession();
		Node userNode = getUserNode(sharedNode);

		//		String folderName = getNodeName(userNode);
		//		String description = getDescription();


		JCRWorkspaceItem itemUnshared = (JCRWorkspaceItem) workspace.getItem(sharedNode.getIdentifier());
		//add UNSHARE operation in History
		itemUnshared.setUnshareHistory(workspace.getOwner().getPortalLogin());
		session.save();		


		//propagade unshare history to parent
		//		JCRWorkspaceItem parentItemUnshared = null;
		//		try{
		//			parentItemUnshared = (JCRWorkspaceItem) workspace.getItem(sharedNode.getParent().getIdentifier());
		//			//add UNSHARE operation in History
		//			parentItemUnshared.setUnshareHistory(workspace.getOwner().getPortalLogin());
		//			session.save();		
		//		}catch (Exception e) {
		//			// TODO: handle exception
		//		}
		//change owner
		//		itemUnshared.setOwnerToCurrentUser(itemUnshared);


		// Check if it's possible remove shared node completely
		checkRemoveSharedFolder(sharedNode);

		// shareNode parent it's the same of destinationNode
		if (destinationNode.getIdentifier().equals(userNode.getParent().getIdentifier())) {
			removeUserSharedFolder(sharedNode);
			session.save();
		}

		//		String unSharedFolderId = workspace.createFolder(folderName, description, destinationNode.getIdentifier()).getId();
		//
		//
		//		Node nodeFolder = session.getNodeByIdentifier(unSharedFolderId);
		//
		//		for (NodeIterator iterator = sharedNode.getNodes(); iterator.hasNext();) {
		//			Node child = (Node) iterator.next();
		//
		//			if (!child.getName().startsWith(JCRRepository.HL_NAMESPACE) 
		//					&& !child.getName().startsWith(JCRRepository.JCR_NAMESPACE)) {			
		//
		//				session.getWorkspace().copy(child.getPath(), nodeFolder.getPath() 
		//						+ workspace.getPathSeparator() + child.getName());
		//			}
		//		}


		//			session.getWorkspace().copy(sharedNode.getPath(), destinationNode.getPath() 
		//					+ workspace.getPathSeparator() + sharedNode.getName());

		session.save();
		//TODO temporarily 

		//		logger.debug("copyremotecontent from "+  nodeFolder.getPath() + " to " + destinationNode.getPath());
		//		try{
		//			workspace.copyRemoteContent(nodeFolder, destinationNode);
		//		}catch (Exception e) {
		//			// TODO: handle exception
		//		}


		//		session.save();		


		return destinationNode;
	}



	@Override
	public Node internalCopy(Node nodeFolder, String newName) throws InternalErrorException,
	ItemAlreadyExistException, WrongDestinationException, RepositoryException{

		Session session = nodeFolder.getSession();

		Node node = session.getNodeByIdentifier(identifier);

		String pathNewNode = nodeFolder.getPath()
				+ workspace.getPathSeparator() + Text.escapeIllegalJcrChars(newName);

		try {

			if(session.getNode(pathNewNode) != null)
				throw new ItemAlreadyExistException(newName + " already exist");
		} catch(RepositoryException e) {

		}

		try {

			Node newNode = unShareNode(node, nodeFolder);

			newNode.setProperty(LAST_MODIFIED,Calendar.getInstance());
			newNode.setProperty(LAST_MODIFIED_BY,workspace.getOwner().getPortalLogin());
			newNode.setProperty(TITLE, newName);
			newNode.setProperty(LAST_ACTION,WorkspaceItemAction.CLONED.toString());
			newNode.getSession().save();
			return newNode;
		} catch (ItemExistsException e) {
			throw new ItemAlreadyExistException(e.getMessage());
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} 
	}

	@Override
	public WorkspaceFolder getParent() throws InternalErrorException {

		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return workspace.getParent(getUserNode(node));
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public String getPath(Node node) throws RepositoryException, InternalErrorException {

		Node userNode =  getUserNode(node);
		return ((JCRWorkspaceFolder)getParent(userNode)).getPath(userNode.getParent())
				+ workspace.getPathSeparator() + getNodeName(userNode);		
	}

	private void removeUserSharedFolder(Node sharedNode) throws InternalErrorException {

		try {
			Session session = sharedNode.getSession();
			Node userNode = getUserNode(sharedNode);

			// Remove user in sharingSet
			Node usersNode = sharedNode.getNode(USERS);
			usersNode.getProperty(workspace.getOwner().getPortalLogin()).remove();

			// Remove sharedNode from user workspace
			session.removeItem(userNode.getPath());
			session.save();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		}
	}

	private void checkRemoveSharedFolder(Node sharedNode) throws  InternalErrorException, RemoteBackendException {

		if (getUsers().isEmpty()) {

			try {
				//TODO temporarily solution
				try{
					workspace.removeRemoteContent(sharedNode, null);
				}catch (Exception e) {
					logger.warn(sharedNode.getName() + " payload not available", e);
				}

				sharedNode.remove();
				sharedNode.getSession().save();
			} catch (RepositoryException e) {
				throw new InternalErrorException(e);
			}
		}

	}


	@Override
	public void remove() throws InternalErrorException,
	InsufficientPrivilegesException {

		logger.debug("remove shared");
		logger.debug("portalLogin: " + JCRRepository.portalLogin + " - owner: "+ getOwner().getPortalLogin());
		Session session = JCRRepository.getSession();
		try {

			Node sharedNode = session.getNodeByIdentifier(getId());
			
			if (!JCRPrivilegesInfo.canDelete(getOwner().getPortalLogin(), JCRRepository.portalLogin, getSharePath(), true)) 
				throw new InsufficientPrivilegesException("Insufficient Privileges to remove the node");

			removeUserSharedFolder(sharedNode);
			checkRemoveSharedFolder(sharedNode);
			//			System.out.println(getSharePath() + " is shared and can be deleted by " + JCRRepository.portalLogin);
			//			System.out.println("remove");

		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (RemoteBackendException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public List<String> getUsers() throws InternalErrorException {

		ArrayList<String> list = new ArrayList<String>();
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(getId());
			Node usersNode = node.getNode(USERS);
			for (PropertyIterator iterator = usersNode.getProperties(); iterator.hasNext();) {
				Property property  = iterator.nextProperty();
				String name = property.getName();
				if (!name.startsWith(JCRRepository.JCR_NAMESPACE) &&
						!name.startsWith(JCRRepository.HL_NAMESPACE))
					list.add(name);
			} 
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		return list;
	}

	private void addUser(Node sharedNode, String user, String destinationFolderId) throws InternalErrorException {

		try {
			Session	session = sharedNode.getSession();

			HomeManager homeManager = workspace.getHome().getHomeManager();
			WorkspaceFolder userRoot = (WorkspaceFolder)homeManager.getHome(user).getWorkspace().getItem(destinationFolderId);

			//			System.out.println("userRoot " + userRoot.getPath());
			Node rootNode =  session.getNodeByIdentifier(userRoot.getId());


			String sharedFolderName = WorkspaceUtil.getUniqueName(
					sharedNode.getProperty(TITLE).getString(),
					userRoot);

			String pathUser = null;

			Node usersNode = sharedNode.getNode(USERS);

			if (applicationName != null){
				pathUser = rootNode.getPath() + workspace.getPathSeparator() + sharedFolderName;
				//				itemName
				session.getWorkspace().clone(session.getWorkspace().getName(),
						sharedNode.getPath()+ workspace.getPathSeparator() + itemName, pathUser, false);

				logger.trace("clone from " + sharedNode.getPath()+ workspace.getPathSeparator() + itemName + " to "+ pathUser);

				session.save();

			}else {			
				pathUser = rootNode.getPath() + workspace.getPathSeparator() + sharedFolderName;
				try {
					if (usersNode.getProperty(user) != null)
						return;
				} catch (PathNotFoundException e) {
					logger.debug("User is not present");
				}
				session.getWorkspace().clone(session.getWorkspace().getName(),
						sharedNode.getPath(), pathUser, false);

				session.save();

			}

			String value = userRoot.getId() + workspace.getPathSeparator() 
					+ sharedFolderName;
			usersNode.setProperty(user, value);
			session.save();


		} catch (Exception e) {
			e.printStackTrace();
			throw new InternalErrorException(e);
		}

	}

	@Override
	public void addUser(String user) throws InsufficientPrivilegesException,
	InternalErrorException {

		Session session = JCRRepository.getSession(); 
		try {	 
			Node sharedNode = session.getNodeByIdentifier(getId());
			Node usersNode = sharedNode.getNode(USERS);

			try {
				if (usersNode.getProperty(user) != null)
					return;
			} catch (PathNotFoundException e) {
				logger.debug("User "+ user + " is not present");
			}

			HomeManager homeManager = workspace.getHome().getHomeManager();
			//GCUBEScope scopeOwner = workspace.getOwner().getScope();
			Home home = homeManager.getHome(user);

			addUser(sharedNode,user, home.getWorkspace().getRoot().getId());


		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public WorkspaceItemType getType() {
		return WorkspaceItemType.SHARED_FOLDER;
	}

	@Override
	public WorkspaceFolder unShare() throws InternalErrorException {

		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);	
			Node userNode = getUserNode(node);			

			//			Node unsharedNode = unShareNode(node, userNode.getParent());
			//			Node rootNode = unShareNode(node, userNode.getParent());

			unShareNode(node, userNode.getParent());

			//			return new JCRWorkspaceFolder(workspace, unsharedNode);
			return null;
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}

	}

	@Override
	public WorkspaceSharedFolder share(List<String> users) throws InsufficientPrivilegesException,
	WrongDestinationException, InternalErrorException {

		for (String user : users) {
			addUser(user);
		}

		//set history
		setShareHistory(users, workspace.getOwner().getPortalLogin());

		return this;
	}



	@Override
	public String getName(String user) throws InternalErrorException {

		Session session = JCRRepository.getSession();
		try {
			Node sharedNode = session.getNodeByIdentifier(getId());
			Node userNode = getUserNode(sharedNode, user);
			return getNodeName(userNode);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}	
	}

	//get path in root folder /Share (e.g. /Share/52fb2641-c9ad-4008-8205-8ea3359dc271)
	public String getSharePath() throws InternalErrorException{
		Session session = JCRRepository.getSession();
		try {
			return session.getNodeByIdentifier(identifier).getPath();
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public void setACL(List<String> users, ACLType privilege) throws InternalErrorException {

		String absPath = null;
		JCRAccessManager accessManager = new JCRAccessManager();
		try{

			absPath = getSharePath();

			switch(privilege){

			case READ_ONLY:
				System.out.println("READ_ONLY");
				accessManager.setReadOnlyACL(users, absPath);		
				break;
			case WRITE_OWNER:	
				System.out.println("WRITE_OWNER");
				accessManager.setWriteOwnerACL(users, absPath);		
				break;
			case WRITE_ALL:
				System.out.println("WRITE_ALL");
				accessManager.setWriteAllACL(users, absPath);	
				break;
			case ADMINISTRATOR:
				System.out.println("ADMINISTRATOR");
				accessManager.setAdminACL(users, absPath);			
				break;
			default:
				break;
			}

		}catch (Exception e) {
			logger.error("an error occurred setting ACL on: " + absPath);
		}

	}

}
