package org.gcube.event.publisher;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractEventPublisher implements EventPublisher {

    protected static final Logger logger = LoggerFactory.getLogger(AbstractEventPublisher.class);

    private EventSender eventSender;
    private ResultsParser resultsParser;

    public AbstractEventPublisher() {
        this.eventSender = createEventSender();
        this.resultsParser = createResultsParser();
    }

    @Override
    public void publish(Event event) {
        publish(event, false);
    }

    @Override
    public String publish(Event event, boolean waitForResult) {
        if (event != null) {
            if (waitForResult) {
                return getEventSender().sendAndGetResult(event);
            } else {
                getEventSender().send(event);
            }
        } else {
            logger.warn("Cannot publish a null event");
        }
        return null;
    }

    @Override
    public EventStatus check(String instanceId) {
        if (instanceId != null) {
            return getResultsParser().parseResults(eventSender.retrive(instanceId));
        } else {
            logger.warn("Cannot check with a null instance ID");
            return EventStatus.NOT_FOUND();
        }
    }

    protected abstract EventSender createEventSender();

    protected ResultsParser createResultsParser() {
        return new ConductorResultsParser();
    }

    public EventSender getEventSender() {
        return eventSender;
    }

    public void setEventSender(EventSender eventSender) {
        this.eventSender = eventSender;
    }

    public ResultsParser getResultsParser() {
        return resultsParser;
    }

    public void setResultsParser(ResultsParser resultsParser) {
        this.resultsParser = resultsParser;
    }

}
