package org.gcube.common.iam;

import java.util.List;

import org.gcube.common.keycloak.KeycloakClient;
import org.gcube.common.keycloak.KeycloakClientException;
import org.gcube.common.keycloak.model.TokenResponse;

/**
 * D4Science IAM client authorization implementation.
 * Handles UMA (User-Managed Access) token requests for authorization with context and permissions.
 * 
 * @author Mauro Mugnaini (Nubisware S.r.l.)
 */
public class D4ScienceIAMClientAuthz extends AbstractIAMResponse {

    /**
     * Creates a new authorization instance using an existing authentication.
     * 
     * @param authn the authentication instance
     * @param context the context for authorization
     * @param permissions the list of permissions to request
     * @throws D4ScienceIAMClientException if authorization fails
     */
    protected D4ScienceIAMClientAuthz(D4ScienceIAMClientAuthn authn, String context, List<String> permissions)
            throws D4ScienceIAMClientException {

        super(authn.getIamClient(), performAuthz(authn.getIamClient(), authn.getTokenResponse(), context, permissions));
    }

    /**
     * Performs authorization using an existing authentication token.
     * 
     * @param iamClient the IAM client
     * @param authnTR the authentication token response
     * @param context the context for authorization
     * @param permissions the list of permissions to request
     * @return the authorization token response
     * @throws D4ScienceIAMClientException if authorization fails
     */
    private static final TokenResponse performAuthz(D4ScienceIAMClient iamClient, TokenResponse authnTR,
            String context, List<String> permissions) throws D4ScienceIAMClientException {

        KeycloakClient keycloakClient = iamClient.getKeycloakClient();
        try {
            return keycloakClient.queryUMAToken(keycloakClient.getTokenEndpointURL(iamClient.getRealmBaseURL()),
                    authnTR, context, permissions);

        } catch (KeycloakClientException e) {
            throw new D4ScienceIAMClientException(e);
        }
    }

    /**
     * Creates a new authorization instance using client credentials.
     * 
     * @param iamClient the IAM client
     * @param clientId the client ID
     * @param clientSecret the client secret
     * @param context the context for authorization
     * @param permissions the list of permissions to request
     * @throws D4ScienceIAMClientException if authorization fails
     */
    protected D4ScienceIAMClientAuthz(D4ScienceIAMClient iamClient, String clientId, String clientSecret,
            String context, List<String> permissions) throws D4ScienceIAMClientException {

        super(iamClient, performAuthz(iamClient, clientId, clientSecret, context, permissions));
    }

    /**
     * Performs authorization using client credentials.
     * 
     * @param iamClient the IAM client
     * @param clientId the client ID
     * @param clientSecret the client secret
     * @param context the context for authorization
     * @param permissions the list of permissions to request
     * @return the authorization token response
     * @throws D4ScienceIAMClientException if authorization fails
     */
    private static final TokenResponse performAuthz(D4ScienceIAMClient iamClient, String clientId, String clientSecret,
            String context, List<String> permissions) throws D4ScienceIAMClientException {

        KeycloakClient keycloakClient = iamClient.getKeycloakClient();
        try {
            return keycloakClient.queryUMAToken(keycloakClient.getTokenEndpointURL(iamClient.getRealmBaseURL()),
                    clientId, clientSecret, context, permissions);

        } catch (KeycloakClientException e) {
            throw new D4ScienceIAMClientException(e);
        }
    }

}