package org.gcube.common.security;

import java.util.ArrayList;
import java.util.List;

/**
 * Represents the owner of a resource or process. An owner can be either a human user
 * or an application. This class stores various attributes of the owner, including their
 * identity, roles, and contact information.
 */
public class Owner {

	private String id;

	private List<String> roles = new ArrayList<String>();
	private List<String> globalRoles = new ArrayList<String>();
	private List<String> contextRoles = new ArrayList<String>();

	private String email;
	private String firstName;
	private String lastName;
	
	private boolean externalClient;

	// "name", as the client pretty name, e.g. Catalogue for gcat, Workspace for storage-hub
	private String clientName;

	// username of the user who requested such client
	private String contactPerson;

	//the name of the organisation / community using such client. D4Science will be used for internal clients.
	private String contactOrganisation;		
		
	private boolean application;
	
	/**
	 * Constructs an {@code Owner} object for a user or application with a set of context-specific roles.
	 *
	 * @param id The unique identifier of the owner.
	 * @param contextRoles A list of roles specific to the current context.
	 * @param external A boolean flag indicating if the owner is an external client.
	 * @param application A boolean flag indicating if the owner is an application rather than a human user.
	 */
	public Owner(String id, List<String> contextRoles, boolean external, boolean application) {
		super();
		this.id = id;
		this.contextRoles = contextRoles;
		this.roles.addAll(contextRoles);
		this.externalClient = external;
		this.application = application;
	}
	
	/**
	 * Constructs an {@code Owner} object for a human user with context-specific roles and personal details.
	 *
	 * @param id The unique identifier of the owner.
	 * @param contextRoles A list of roles specific to the current context.
	 * @param email The email address of the owner.
	 * @param firstName The first name of the owner.
	 * @param lastName The last name of the owner.
	 * @param external A boolean flag indicating if the owner is an external client.
	 * @param application A boolean flag indicating if the owner is an application.
	 */
	public Owner(String id, List<String> contextRoles, String email, String firstName, String lastName, boolean external, boolean application) {
		this(id, contextRoles, external, application);
		this.email = email;
		this.firstName = firstName;
		this.lastName = lastName;
	}
	
	/**
	 * Constructs an {@code Owner} object with both context-specific and global roles, along with personal details.
	 *
	 * @param id The unique identifier of the owner.
	 * @param contextRoles A list of roles specific to the current context.
	 * @param globalRoles A list of roles that apply globally.
	 * @param email The email address of the owner.
	 * @param firstName The first name of the owner.
	 * @param lastName The last name of the owner.
	 * @param external A boolean flag indicating if the owner is an external client.
	 * @param application A boolean flag indicating if the owner is an application.
	 */
	public Owner(String id, List<String> contextRoles, List<String> globalRoles, String email, String firstName, String lastName, boolean external, boolean application) {
		this(id, contextRoles, email, firstName, lastName, external, application);
		this.globalRoles = globalRoles;
		this.roles.addAll(globalRoles);
	}

	/**
	 * Checks if the owner is an application rather than a human user.
	 *
	 * @return {@code true} if the owner is an application, {@code false} otherwise.
	 */
	public boolean isApplication() {
		return application;
	}
	
	/**
	 * Gets the unique identifier of the owner.
	 *
	 * @return The owner's unique ID.
	 */
	public String getId() {
		return id;
	}

	/**
	 * Returns all roles assigned to the owner, including both global and context-specific roles.
	 *
	 * @return A list of all roles.
	 */
	public List<String> getRoles() {
		return roles;
	}
	
	/**
	 * Returns the roles of the owner that are not specific to a context.
	 *
	 * @return A list of the global roles.
	 */
	public List<String> getGlobalRoles() {
		return globalRoles;
	}

	/**
	 * Returns the roles of the owner that are specific to a particular context.
	 *
	 * @return A list of the context-specific roles.
	 */
	public List<String> getContextRoles() {
		return contextRoles;
	}
	
	/**
	 * Gets the email address of the owner.
	 *
	 * @return The owner's email address.
	 */
	public String getEmail() {
		return email;
	}
	
	/**
	 * Gets the first name of the owner.
	 *
	 * @return The owner's first name.
	 */
	public String getFirstName() {
		return firstName;
	}
	
	/**
	 * Gets the last name of the owner.
	 *
	 * @return The owner's last name.
	 */
	public String getLastName() {
		return lastName;
	}

	/**
	 * Checks if the owner is an external client.
	 *
	 * @return {@code true} if the owner is an external client, {@code false} otherwise.
	 */
	public boolean isExternalClient() {
		return externalClient;
	}

	/**
	 * Gets the pretty name of the client application, if applicable (e.g., "Catalogue").
	 *
	 * @return The client's name.
	 */
	public String getClientName() {
		return clientName;
	}

	/**
	 * Sets the pretty name of the client application.
	 *
	 * @param clientName The name to set for the client.
	 */
	public void setClientName(String clientName) {
		this.clientName = clientName;
	}

	/**
	 * Gets the username of the user who requested the client application.
	 *
	 * @return The username of the contact person.
	 */
	public String getContactPerson() {
		return contactPerson;
	}

	/**
	 * Sets the username of the user who requested the client application.
	 *
	 * @param contactPerson The username to set for the contact person.
	 */
	public void setContactPerson(String contactPerson) {
		this.contactPerson = contactPerson;
	}

	/**
	 * Gets the name of the organization or community using the client.
	 *
	 * @return The name of the contact organization.
	 */
	public String getContactOrganisation() {
		return contactOrganisation;
	}

	/**
	 * Sets the name of the organization or community using the client.
	 *
	 * @param contactOrganisation The name to set for the contact organization.
	 */
	public void setContactOrganisation(String contactOrganisation) {
		this.contactOrganisation = contactOrganisation;
	}
	
}