package org.gcube.application.shlink.discovery;

/**
 * Bean class representing the configuration for the URL shortener service. This
 * class holds the essential properties needed to interact with the Shlink URL
 * shortener API.
 * 
 * @author Francesco Mangiacrapa at ISTI-CNR
 * @since July 2025
 */
public class UrlShortenerConfig {

	/**
	 * The URL of the Shlink service endpoint
	 */
	private String shortnerServiceUrl;

	/**
	 * The decrypted API key for authentication with the Shlink service
	 */
	private String apiKey;

	/**
	 * The domain to use for generating short URLs (optional)
	 */
	private String domain;

	/**
	 * Default constructor.
	 */
	public UrlShortenerConfig() {
		// Default constructor
	}

	/**
	 * Constructor with all parameters.
	 *
	 * @param shortnerServiceUrl the service URL
	 * @param apiKey             the api key
	 * @param domain             the domain (can be null)
	 */
	public UrlShortenerConfig(String shortnerServiceUrl, String apiKey, String domain) {
		this.shortnerServiceUrl = shortnerServiceUrl;
		this.apiKey = apiKey;
		this.domain = domain;
	}

	/**
	 * Gets the service URL.
	 *
	 * @return the service URL
	 */
	public String getShortnerServiceUrl() {
		return shortnerServiceUrl;
	}

	/**
	 * Sets the service URL.
	 *
	 * @param shortnerServiceUrl the service URL to set
	 */
	protected void setShortnerServiceUrl(String shortnerServiceUrl) {
		this.shortnerServiceUrl = shortnerServiceUrl;
	}

	/**
	 * Gets the domain.
	 *
	 * @return the domain
	 */
	public String getDomain() {
		return domain;
	}

	/**
	 * Gets the api key.
	 *
	 * @return the api key
	 */
	public String getApiKey() {
		return apiKey;
	}

	/**
	 * Sets the api key.
	 *
	 * @param apiKey the new api key
	 */
	protected void setApiKey(String apiKey) {
		this.apiKey = apiKey;
	}

	/**
	 * Sets the domain.
	 *
	 * @param domain the domain to set
	 */
	protected void setDomain(String domain) {
		this.domain = domain;
	}

	/**
	 * Validates that the configuration has the required properties.
	 *
	 * @return true if the configuration is valid, false otherwise
	 */
	public boolean isValid() {
		return shortnerServiceUrl != null && !shortnerServiceUrl.trim().isEmpty() && apiKey != null
				&& !apiKey.trim().isEmpty();
	}

	/**
	 * To string.
	 *
	 * @return the string
	 */
	@Override
	public String toString() {
		return "UrlShortenerConfig{" + "shortnerServiceUrl='" + shortnerServiceUrl + '\'' + ", apiKey='"
				+ (apiKey != null ? apiKey.substring(0, Math.min(10, apiKey.length())) + "***MASKED***" : "null") + '\''
				+ ", domain='" + domain + '\'' + '}';
	}
}
