package org.gcube.application.shlink;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;

import org.gcube.application.shlink.discovery.FetchUrlShortenerFromNewIS;
import org.gcube.application.shlink.discovery.FetchUrlShortenerFromOldIS;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.LoggerFactory;

/**
 * A URL shortener implementation that uses the Shlink REST API service.
 *
 * <p>
 * This class provides functionality to create shortened URLs by communicating
 * with a Shlink server instance through its REST API. Shlink is an open-source
 * URL shortener that can be self-hosted.
 * </p>
 *
 * <h2>Usage Example:</h2>
 * 
 * <pre>
 * ShlinkShortener shortener = new ShlinkShortener();
 * String shortUrl = shortener.createShortUrl("https://your-shlink-instance.com/rest/v3/short-urls", "your-api-key",
 * 		"https://example.com/very-long-url");
 * </pre>
 *
 * <h2>Requirements:</h2>
 * <ul>
 * <li>A running Shlink server instance with REST API enabled</li>
 * <li>Valid API key with permissions to create short URLs</li>
 * <li>Network connectivity to the Shlink server</li>
 * </ul>
 * ...
 * 
 * @author Francesco Mangiacrapa
 * 
 * @see <a href="https://shlink.io/">Shlink Official Website</a>
 * @see <a href="https://api-spec.shlink.io/">Shlink API Documentation</a>
 */
public class ShlinkShortener {

	private static final String TAG_KEYWORDS_PROPERTIES = "tag-keywords.properties";

	/** The Resurce Name registered in the Information System */
	public static final String SHORTENER = "HTTP-URL-Shortener-Shlink";
	/** Property name for API key in the resource */
	public static final String KEY = "API_KEY";
	public static final String DOMAIN = "DOMAIN";

	/**
	 * Default tag used when no keywords are found in the URL
	 */
	private static final String DEFAULT_TAG = "workspace";

	protected static org.slf4j.Logger LOGGER = LoggerFactory.getLogger(ShlinkShortener.class);

	/**
	 * Default keywords used to automatically tag URLs based on their content.
	 * Keywords are loaded from the tag-keywords.properties file.
	 */
	private static final String[] TAG_KEYWORDS = loadTagKeywords();

	private boolean isAvailable;

	private FetchUrlShortenerResource fetchUrlShortener;

	public ShlinkShortener(String scope) throws Exception {
		LOGGER.info("Initialized ShlinkShortener with keywords: " + java.util.Arrays.toString(TAG_KEYWORDS) );
		fetchUrlShortener(scope);
		LOGGER.info("ShlinkShortener is available: " + isAvailable);
	}

	/**
	 * Fetches the URL shortener configuration from the runtime resource in the given scope.
	 * First tries to fetch from the New Information System, if not available,
	 * tries to fetch from the Old Information System.
	 * @param scope
	 * @throws Exception
	 */
	private void fetchUrlShortener(String scope) throws Exception {
		try {
			LOGGER.info("Trying to read the Runtime Resource with name {}, in the scope: {}", SHORTENER, scope);

			if (scope == null || scope.isEmpty()) {
				String msg = "Scope is null or empty!";
				throw new Exception(msg);
			}
			// Fetch the URL shortener service configuration from the runtime resource
			fetchUrlShortener = new FetchUrlShortenerFromNewIS(scope);
			isAvailable = fetchUrlShortener.isAvailable();
			if (!isAvailable) {
				LOGGER.warn("The URL Shortener from New IS is not available, trying to fetch from Old IS");
				fetchUrlShortener = new FetchUrlShortenerFromOldIS(scope);
				isAvailable = fetchUrlShortener.isAvailable();
			}

			fetchUrlShortener = new FetchUrlShortenerFromOldIS(scope);
			isAvailable = fetchUrlShortener.isAvailable();
		} catch (Exception e) {
			LOGGER.error(
					"An error occurred reading Runtime Resource for name: " + SHORTENER + ", the scope is: " + scope,
					e);
			isAvailable = false;
			throw new Exception("No " + SHORTENER + " available!");
		}
	}

	/**
	 * Loads tag keywords from the properties file.
	 * 
	 * @return array of keywords loaded from properties file, or default keywords if
	 *         file cannot be read
	 */
	private static String[] loadTagKeywords() {
		java.util.Properties props = new java.util.Properties();
		java.util.List<String> keywords = new java.util.ArrayList<>();

		try (InputStream input = ShlinkShortener.class.getClassLoader().getResourceAsStream(TAG_KEYWORDS_PROPERTIES)) {

			if (input == null) {
				LOGGER.warn(TAG_KEYWORDS_PROPERTIES + " file not found, using default keywords");
				return getDefaultKeywords();
			}

			props.load(input);

			// Load keywords in order (tag.keyword.1, tag.keyword.2, etc.)
			int index = 1;
			while (true) {
				String keyword = props.getProperty("tag.keyword." + index);
				if (keyword == null || keyword.trim().isEmpty()) {
					break;
				}
				keywords.add(keyword.trim());
				index++;
			}

			if (keywords.isEmpty()) {
				LOGGER.warn("No keywords found in properties file, using defaults");
				return getDefaultKeywords();
			}

			LOGGER.info("Loaded " + keywords.size() + " tag keywords from properties file");
			return keywords.toArray(new String[0]);

		} catch (IOException e) {
			LOGGER.error("Error loading tag keywords from " + TAG_KEYWORDS_PROPERTIES + " properties file", e);
			return getDefaultKeywords();
		}
	}

	/**
	 * Returns default keywords as fallback when properties file cannot be loaded.
	 * 
	 * @return array of default keywords
	 */
	private static String[] getDefaultKeywords() {
		return new String[] { "shub", "workspace-explorer-app", "gis", "analytics", "knime", "wekeo", "geo", "ctlg",
				"smp" };
	}

	/**
	 * Extracts tags from a long URL based on predefined keywords. If no keywords
	 * are found, returns the default "workspace" tag.
	 * 
	 * @param longUrl the URL to analyze for keywords
	 * @return array of tags extracted from the URL, or default tag if none found
	 */
	private String[] extractTags(String longUrl) {
		if (longUrl == null || longUrl.trim().isEmpty()) {
			return new String[] { DEFAULT_TAG };
		}

		java.util.List<String> foundTags = new java.util.ArrayList<>();
		String lowerUrl = longUrl.toLowerCase();

		for (String keyword : TAG_KEYWORDS) {
			if (lowerUrl.contains(keyword)) {
				foundTags.add(keyword);
			}
		}

		if (foundTags.isEmpty()) {
			foundTags.add(DEFAULT_TAG);
		}

		LOGGER.debug("Extracted tags from URL: " + foundTags);
		return foundTags.toArray(new String[0]);
	}

	/**
	 * Combines default tags extracted from URL with additional user-provided tags.
	 * Removes duplicates and maintains order (default tags first, then user tags).
	 * 
	 * @param longUrl        the URL to extract default tags from
	 * @param additionalTags additional tags provided by user (can be null)
	 * @return combined array of unique tags
	 */
	private String[] combineTags(String longUrl, String[] additionalTags) {
		String[] defaultTags = extractTags(longUrl);

		if (additionalTags == null || additionalTags.length == 0) {
			return defaultTags;
		}

		java.util.Set<String> uniqueTags = new java.util.LinkedHashSet<>();

		// Add default tags first
		for (String tag : defaultTags) {
			if (tag != null && !tag.trim().isEmpty()) {
				uniqueTags.add(tag.trim());
			}
		}

		// Add additional tags
		for (String tag : additionalTags) {
			if (tag != null && !tag.trim().isEmpty()) {
				uniqueTags.add(tag.trim());
			}
		}

		String[] result = uniqueTags.toArray(new String[0]);
		LOGGER.debug("Combined tags: " + java.util.Arrays.toString(result));
		return result;
	}
	
	/**
	 * Shorten URL with domain and additional tags.
	 *
	 * @param longUrl        the long url
	 * @param additionalTags optional additional tags to associate with the short
	 *                       URL (combined with auto-extracted tags)
	 * @param findIfExists if true, checks if the short URL already exists before creating a new one
	 * @return a shorten url
	 * @throws Exception the exception
	 */
	public String shorten(String longUrl, String[] additionalTags, boolean findIfExists) throws Exception {

		if (longUrl == null) {
			return longUrl;
		}

		String domain = fetchUrlShortener.getDomain();
		String shlikServiceUrl = fetchUrlShortener.getShortnerServiceUrl();
		String apiKey = fetchUrlShortener.getApiKey();

		LOGGER.debug("shorten the input longUrl: " + longUrl + ", with domain: " + domain + ", and additional tags: "
				+ (additionalTags != null ? java.util.Arrays.toString(additionalTags) : "none") + ", findIfExists: " + findIfExists);

		if (!isAvailable) {
			LOGGER.warn("The URL Shortener is not available, returning the original long URL");
			return longUrl;
		}

		ShlinkRequest request = new ShlinkRequest.Builder()
				.shlinkBaseUrl(shlikServiceUrl)
				.shlinkAPIKey(apiKey)
				.longUrl(longUrl)
				.domain(domain)
				.additionalTags(additionalTags)
				.findIfExists(findIfExists)
				.build();
		
		String shortUrl = this.createShortUrl(request);
		if (shortUrl != null) {
			LOGGER.debug("Shortened URL: " + shortUrl);
			return shortUrl;
		} else {
			LOGGER.error("Failed to create short URL for: " + longUrl);
			return longUrl; // Return the original URL if shortening fails
		}
	}

	/**
	 * Creates a short URL using the Shlink API service. Automatically extracts tags
	 * from the URL based on predefined keywords.
	 * 
	 * @param shlinkBaseUrl the base URL of the Shlink service
	 * @param shlinkAPIKey  the API key for authentication
	 * @param longUrl       the original long URL to be shortened
	 * @return the shortened URL as a String if successful, null if the operation
	 *         fails
	 */
	public String createShortUrl(String shlinkBaseUrl, String shlinkAPIKey, String longUrl) {
		ShlinkRequest shlinkRequest = new ShlinkRequest.Builder().shlinkBaseUrl(shlinkBaseUrl)
				.shlinkAPIKey(shlinkAPIKey).longUrl(longUrl).build();
		return createShortUrl(shlinkRequest);
	}

	private String createShortUrl(ShlinkRequest shlinkRequest) {
		HttpURLConnection connection = null;
		try {
			LOGGER.debug("shlinkBaseUrl: " + shlinkRequest.getShlinkBaseUrl());
			// Construct the full API endpoint URL
			String apiEndpoint = shlinkRequest.getShlinkBaseUrl();
			if (!apiEndpoint.endsWith("/")) {
				apiEndpoint += "/";
			}
			apiEndpoint += "rest/v3/short-urls";

			// Build JSON request with optional domain and combined tags
			String jsonInputString;
			try {
				JSONObject jsonRequest = new JSONObject();
				jsonRequest.put("longUrl", shlinkRequest.getLongUrl());

				if (shlinkRequest.getDomain() != null && !shlinkRequest.getDomain().trim().isEmpty()) {
					jsonRequest.put("domain", shlinkRequest.getDomain().trim());
					LOGGER.info("Using custom domain: " + shlinkRequest.getDomain());
				}

				// Combine auto-extracted tags with additional tags
				String[] finalTags = combineTags(shlinkRequest.getLongUrl(), shlinkRequest.getAdditionalTags());
				if (finalTags != null && finalTags.length > 0) {
					jsonRequest.put("tags", finalTags);
					LOGGER.info("Using tags: " + java.util.Arrays.toString(finalTags));
				}

				// Add findIfExists parameter
				if (shlinkRequest.isFindIfExists()) {
					jsonRequest.put("findIfExists", true);
					LOGGER.info("Using findIfExists: true");
				}

				jsonInputString = jsonRequest.toString();
			} catch (JSONException e) {
				LOGGER.error("Failed to build JSON request", e);
				return null;
			}

			LOGGER.info("Request JSON: " + jsonInputString);

			URL url = new URL(apiEndpoint);
			connection = (HttpURLConnection) url.openConnection();
			connection.setRequestMethod("POST");
			connection.setRequestProperty("X-Api-Key", shlinkRequest.getShlinkAPIKey());
			connection.setRequestProperty("Content-Type", "application/json");
			connection.setRequestProperty("Accept", "application/json");
			connection.setDoOutput(true);

			LOGGER.debug("Sending POST request to: " + apiEndpoint);

			try (OutputStream os = connection.getOutputStream()) {
				byte[] input = jsonInputString.getBytes(StandardCharsets.UTF_8);
				os.write(input, 0, input.length);
			}

			int responseCode = connection.getResponseCode();
			LOGGER.debug("Response Code: " + responseCode);

			InputStream inputStream;
			if (responseCode >= 200 && responseCode < 300) {
				inputStream = connection.getInputStream();
			} else {
				inputStream = connection.getErrorStream();
			}

			StringBuilder response = new StringBuilder();
			try (BufferedReader br = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8))) {

				String line;
				while ((line = br.readLine()) != null) {
					response.append(line.trim());
				}
			}

			LOGGER.debug("Response Body: " + response);

			if (responseCode >= 200 && responseCode < 300) {
				JSONObject jsonObject;
				try {
					jsonObject = new JSONObject(response.toString());
					String shortUrl = jsonObject.getString("shortUrl");
					LOGGER.info("Short URL created: " + shortUrl);
					return shortUrl;
				} catch (JSONException e) {
					LOGGER.error("Failed to parse JSON response: " + response, e);
					return null;
				}

			} else {
				LOGGER.error("Request failed! Response: " + response);
				return null;
			}

		} catch (IOException e) {
			LOGGER.error("Exception occurred while creating short URL", e);
		} finally {
			if (connection != null) {
				connection.disconnect();
			}
		}
		return null;
	}
}