package org.gcube.application.shlink;

/**
 * Creates a short URL using the Shlink API service with a specific domain.
 * Automatically extracts tags from the URL and combines them with additional tags if provided.
 * 
 * @param shlinkBaseUrl the base URL of the Shlink service
 * @param shlinkAPIKey the API key for authentication
 * @param longUrl the original long URL to be shortened
 * @param domain the domain to use for the short URL (can be null for default)
 * @param additionalTags additional tags to add to the auto-extracted ones (can be null)
 * @param findIfExists if true, checks if the short URL already exists before creating a new one
 * @throws Exception if an error occurs during the request or response handling
 * @return the shortened URL as a String if successful, null if the operation fails
 */

/**
 * Builder class for creating ShlinkShortener requests with fluent interface.
 * This class represents a request to create a short URL using the Shlink API service.
 * It automatically extracts tags from the URL and combines them with additional tags if provided.
 * 
 * <p>Usage example:</p>
 * <pre>
 * ShlinkRequest request = new ShlinkRequest.Builder()
 *     .shlinkBaseUrl("https://shlink.example.com")
 *     .shlinkAPIKey("your-api-key")
 *     .longUrl("https://example.com/very/long/url")
 *     .domain("short.example.com")
 *     .additionalTags(new String[]{"tag1", "tag2"})
 *     .findIfExists(true)
 *     .build();
 * </pre>
 * 
 * @author Francesco Mangiacrapa
 * @see ShlinkRequest.Builder
 */
public class ShlinkRequest {
    private String shlinkBaseUrl;
    private String shlinkAPIKey;
    private String longUrl;
    private String domain;
    private String[] additionalTags;
    private boolean findIfExists = false;

    public String getShlinkBaseUrl() {
        return shlinkBaseUrl;
    }

    public String getShlinkAPIKey() {
        return shlinkAPIKey;
    }

    public String getLongUrl() {
        return longUrl;
    }

    public String getDomain() {
        return domain;
    }

    public String[] getAdditionalTags() {
        return additionalTags;
    }

    public boolean isFindIfExists() {
        return findIfExists;
    }

    /**
     * Builder class for creating ShlinkRequest instances with fluent interface.
     */
    public static class Builder {
        private String shlinkBaseUrl;
        private String shlinkAPIKey;
        private String longUrl;
        private String domain;
        private String[] additionalTags;
        private boolean findIfExists = false;

        public Builder shlinkBaseUrl(String shlinkBaseUrl) {
            this.shlinkBaseUrl = shlinkBaseUrl;
            return this;
        }

        public Builder shlinkAPIKey(String shlinkAPIKey) {
            this.shlinkAPIKey = shlinkAPIKey;
            return this;
        }

        public Builder longUrl(String longUrl) {
            this.longUrl = longUrl;
            return this;
        }

        public Builder domain(String domain) {
            this.domain = domain;
            return this;
        }

        public Builder additionalTags(String[] additionalTags) {
            this.additionalTags = additionalTags;
            return this;
        }

        public Builder findIfExists(boolean findIfExists) {
            this.findIfExists = findIfExists;
            return this;
        }

        public ShlinkRequest build() {
            ShlinkRequest request = new ShlinkRequest();
            request.shlinkBaseUrl = this.shlinkBaseUrl;
            request.shlinkAPIKey = this.shlinkAPIKey;
            request.longUrl = this.longUrl;
            request.domain = this.domain;
            request.additionalTags = this.additionalTags;
            request.findIfExists = this.findIfExists;
            return request;
        }
    }

}