package org.gcube.application.shlink.discovery;

import java.io.IOException;
import java.io.InputStream;

import org.gcube.application.shlink.FetchUrlShortenerResource;
import org.gcube.informationsystem.resourceregistry.client.ResourceRegistryClient;
import org.gcube.informationsystem.resourceregistry.client.ResourceRegistryClientFactory;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The Class FetchUrlShortenerFromNewIS.
 *
 * This class extends FetchUrlShortenerResource to provide functionality for
 * fetching and managing the URL shortener service configuration from a runtime
 * resource of the NEW Information System.
 * 
 * @author Francesco Mangiacrapa at ISTI-CNR
 * @since July 2025
 */
public class FetchUrlShortenerFromNewIS extends FetchUrlShortenerResource {

	private static final String GCUBE_URL_SHORTENER_DEFAULT_QUERY_TEMPLATE = "gcube-url-shortener";
	protected static Logger logger = LoggerFactory.getLogger(FetchUrlShortenerFromNewIS.class);

	public FetchUrlShortenerFromNewIS(String scope) throws Exception {
		super(scope);
		fetchUrlShortener();
	}

	protected void fetchUrlShortener() throws Exception {

		try {
			logger.info("Trying to read from new IS the Resource with name {}, in the scope: {}", SHORTENER, scope);

			if (scope == null || scope.isEmpty()) {
				String msg = "Scope is null or empty!";
				throw new Exception(msg);
			}

			String queryTemplateName = readFromQueryTemplate();
			if (queryTemplateName == null || queryTemplateName.isEmpty()) {
				queryTemplateName = GCUBE_URL_SHORTENER_DEFAULT_QUERY_TEMPLATE;
				logger.info("Using default query template name: {}", queryTemplateName);
			}

			logger.info("Query template is: {}", queryTemplateName);

			// Initialize ResourceRegistryClient
			ResourceRegistryClient resourceRegistryClient = ResourceRegistryClientFactory.create();

			// Execute query template
			String jsonString = resourceRegistryClient.runQueryTemplate(queryTemplateName);
			logger.debug("Query template response: {}", jsonString);

			// Check if response is valid and not empty JSON
			if (jsonString != null && !jsonString.trim().isEmpty()) {
				String trimmedResponse = jsonString.trim();

				// Check if it's an empty JSON using helper method
				if (isEmptyJsonResponse(trimmedResponse)) {
					isAvailable = false;
					logger.warn("Query template returned empty JSON: {}", trimmedResponse);
					throw new Exception("No resources found in query template response: " + queryTemplateName);
				}

				// Parse the JSON response to extract configuration
				parseJsonResponse(trimmedResponse);


				logger.info("Successfully retrieved configuration from query template: {}", queryTemplateName);
				logger.info("Endpoint: {}", config.getShortnerServiceUrl());
				logger.info(DOMAIN + ": {}", config.getDomain());
				logger.info(KEY + ": {}***MASKED***",
						config.getApiKey() != null
								? config.getApiKey().substring(0, Math.min(10, config.getApiKey().length()))
								: "null");
				isAvailable = true;
			} else {
				isAvailable = false;
				throw new Exception("Empty response from query template: " + queryTemplateName);
			}

		} catch (Exception e) {
			logger.error("An error occurred fetching shortener configuration from new IS for: " + SHORTENER
					+ ", in the the scope: " + scope, e);
			isAvailable = false;
		}
	}

	public boolean isAvailable() {
		return isAvailable;
	}

	/**
	 * Reads the IS_QUERY_TEMPLATE_SHORTENER configuration from the query_template
	 * properties file.
	 * 
	 * @return the URL shortener configuration as a String
	 * @throws IOException if there's an error reading the file
	 */
	private String readFromQueryTemplate() {
		String queryTemplatePath = "query_template_shortener.properties";
		String propertyName = "IS_QUERY_TEMPLATE_SHORTENER";

		java.util.Properties properties = new java.util.Properties();

		// Try to load from classpath first
		try (InputStream inputStream = this.getClass().getClassLoader().getResourceAsStream(queryTemplatePath)) {
			if (inputStream != null) {
				properties.load(inputStream);
				String value = properties.getProperty(propertyName);
				if (value != null && !value.trim().isEmpty()) {
					return value.trim();
				} else {
					logger.warn("Property " + propertyName + " not found or empty");
					return null;
				}
			}
		} catch (Exception e) {
			logger.warn("Could not load from classpath: " + queryTemplatePath, e);
			return null;
		}

		logger.warn("Could not load from classpath: " + queryTemplatePath);
		return null;
	}

	/**
	 * Checks if a JSON response is empty (empty array, empty object, or minimal
	 * content).
	 * 
	 * @param jsonResponse the JSON response to check
	 * @return true if the response is considered empty, false otherwise
	 */
	private boolean isEmptyJsonResponse(String jsonResponse) {
		if (jsonResponse == null || jsonResponse.trim().isEmpty()) {
			return true;
		}

		String trimmed = jsonResponse.trim();

		// Check for empty JSON array
		if (trimmed.equals("[]")) {
			return true;
		}

		// Check for empty JSON object
		if (trimmed.equals("{}")) {
			return true;
		}

		// Check for minimal JSON (likely empty with whitespace)
		if (trimmed.length() <= 2) {
			return true;
		}

		// Try to parse as JSON to check if array/object is effectively empty
		try {
			if (trimmed.startsWith("[") && trimmed.endsWith("]")) {
				JSONArray jsonArray = new JSONArray(trimmed);
				return jsonArray.length() == 0;
			} else if (trimmed.startsWith("{") && trimmed.endsWith("}")) {
				JSONObject jsonObject = new JSONObject(trimmed);
				return jsonObject.length() == 0;
			}
		} catch (Exception e) {
			// If JSON parsing fails, assume it's not empty (might be malformed but has
			// content)
			logger.debug("Could not parse JSON response, assuming not empty: {}", e.getMessage());
			return false;
		}

		return false;
	}

	/**
	 * Parses the JSON response from the query template to extract configuration
	 * properties.
	 * 
	 * @param jsonResponse the JSON response string to parse
	 * @throws Exception if parsing fails or required properties are missing
	 */
	private void parseJsonResponse(String jsonResponse) throws Exception {
		try {
			JSONArray jsonArray = new JSONArray(jsonResponse);

			if (jsonArray.length() == 0) {
				throw new Exception("JSON array is empty");
			}

			// Get the first configuration object from the array
			JSONObject configObject = jsonArray.getJSONObject(0);

			// Extract required properties
			if (configObject.has("ENDPOINT")) {
				config.setShortnerServiceUrl(configObject.getString("ENDPOINT"));
			} else {
				throw new Exception("Missing required property: ENDPOINT");
			}

			if (configObject.has(KEY)) {
				config.setApiKey(configObject.getString(KEY));
			} else {
				throw new Exception("Missing required property: " + KEY);
			}

			// Domain is optional
			if (configObject.has(DOMAIN)) {
				config.setDomain(configObject.getString(DOMAIN));
			}

			// Validate that we have the essential properties
			if (config.getShortnerServiceUrl() == null || config.getShortnerServiceUrl().trim().isEmpty()) {
				throw new Exception("ENDPOINT property is null or empty");
			}

			if (config.getApiKey() == null || config.getApiKey().trim().isEmpty()) {
				throw new Exception(KEY + " property is null or empty");
			}

			logger.debug("Successfully parsed configuration - Endpoint: {}, Domain: {}, API Key length: {}",
					config.getShortnerServiceUrl(), config.getDomain(),
					config.getApiKey() != null ? config.getApiKey().length() : 0);

		} catch (Exception e) {
			logger.error("Error parsing JSON response: {}", jsonResponse, e);
			throw new Exception("Failed to parse configuration from JSON response: " + e.getMessage());
		}
	}

}
