package org.gcube.application.shlink.discovery;

import java.util.List;
import java.util.Map;

import org.gcube.application.shlink.FetchUrlShortenerResource;
import org.gcube.common.encryption.encrypter.StringEncrypter;
import org.gcube.common.resources.gcore.ServiceEndpoint;
import org.gcube.common.resources.gcore.ServiceEndpoint.AccessPoint;
import org.gcube.common.resources.gcore.ServiceEndpoint.Property;
import org.gcube.resources.discovery.client.api.DiscoveryClient;
import org.gcube.resources.discovery.client.queries.impl.XQuery;
import static org.gcube.resources.discovery.icclient.ICFactory.clientFor;
import static org.gcube.resources.discovery.icclient.ICFactory.queryFor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The Class ISOldFetchUrlShortenerResource.
 *
 * This class extends FetchUrlShortenerResource to provide functionality for fetching
 * and managing the URL shortener service configuration from a runtime resource of the OLD Information System.
 * 
 * @author Francesco Mangiacrapa at ISTI-CNR
 * @since July 2025
 */
public class FetchUrlShortenerFromOldIS extends FetchUrlShortenerResource {

    protected static Logger logger = LoggerFactory.getLogger(FetchUrlShortenerFromOldIS.class);

    public FetchUrlShortenerFromOldIS(String scope) throws Exception {
        super(scope);
        fetchUrlShortener();
    }

	protected void fetchUrlShortener() throws Exception {

		// scope is always the current scope. always from the contstructor

		try {
			logger.info("Trying to read the Runtime Resource with name {}, in the scope: {}", SHORTENER, scope);

			if (scope == null || scope.isEmpty()) {
				String msg = "Scope is null or empty!";
				throw new Exception(msg);
			}

			XQuery query = queryFor(ServiceEndpoint.class);
			query.addCondition("$resource/Profile/Name/text() eq '" + SHORTENER + "'")
					.setResult("$resource/Profile/AccessPoint");
			DiscoveryClient<AccessPoint> client = clientFor(AccessPoint.class);
			List<AccessPoint> endpoints = client.submit(query);

			if (endpoints.size() == 0) {
				isAvailable = false;
				logger.warn("Runtime Resource " + SHORTENER + " not found");
				throw new Exception(
						"No Runtime Resource with name: " + SHORTENER + " is available in the scope: " + scope);
			}

			AccessPoint shortnerAP = endpoints.get(0);

			if (shortnerAP != null) {
				config.setShortnerServiceUrl(shortnerAP.address());
				// READING THE PROPERTIES
				Map<String, Property> propertiesMap = shortnerAP.propertyMap();
				if (propertiesMap == null || propertiesMap.size() == 0) {
					throw new Exception("You need to add the configurations as properties to " + SHORTENER);
				}

				// READING KEY
				String encrKey = readPropertyValue(propertiesMap, KEY, true);
				logger.debug("Read property with name {} and value {}", KEY, encrKey);
				if (encrKey != null) {
					String decryptedKey = StringEncrypter.getEncrypter().decrypt(encrKey);
					config.setApiKey(decryptedKey);
					logger.debug("Decrypted {}: {}+++MASKED-KEY+++", KEY,
							decryptedKey.substring(0, decryptedKey.length() - 20));
				} else {
					logger.warn("The property {} is not available in the Runtime Resource {}", KEY, SHORTENER);
					config.setApiKey(null);
					throw new Exception(
							"The property " + KEY + " is not available in the Runtime Resource " + SHORTENER);
				}

				// READING DOMAIN
				String domainValue = readPropertyValue(propertiesMap, DOMAIN, false);
				config.setDomain(domainValue);
				logger.debug("Read property with name {} and value {}", DOMAIN, domainValue);
			}

			if (config.getShortnerServiceUrl() == null || config.getApiKey() == null) {
				String error = "The access point in the RR: " + SHORTENER + " for scope " + scope
						+ " is not available correctly. Impossible to read the service URL or decrypt the password";
				logger.warn(error);
				throw new Exception(error);
			}

			logger.info("HTTP-URL-Shortener found in scope {}, shortnerServiceUrl: {}.", scope,
					config.getShortnerServiceUrl());
			isAvailable = true;

		} catch (Exception e) {
			logger.error(
					"An error occurred reading Runtime Resource for name: " + SHORTENER + ", the scope is: " + scope,
					e);
			isAvailable = false;
			throw new Exception("No " + SHORTENER + " available!");
		}
	}


    public boolean isAvailable() {
        return isAvailable;
    }
    
}
