package org.gcube.application.enm.service.plugins.comps;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.util.List;
import java.util.UUID;

import org.gcube.application.enm.common.xml.logs.ExperimentLogs;
import org.gcube.application.enm.common.xml.logs.LogType;
import org.gcube.application.enm.common.xml.request.ExperimentRequest;
import org.gcube.application.enm.common.xml.request.RequestType;
import org.gcube.application.enm.common.xml.results.ExperimentResults;
import org.gcube.application.enm.common.xml.results.ResultType;
import org.gcube.application.enm.common.xml.status.ExperimentStatus;
import org.gcube.application.enm.common.xml.status.ObjectFactory;
import org.gcube.application.enm.common.xml.status.StatusType;
import org.gcube.application.enm.service.ExecutionException;
import org.gcube.application.enm.service.ExecutionResource;
import org.gcube.application.enm.service.conn.FtpClient;
import org.gcube.application.enm.service.conn.InformationSystemClient.COMPSsCredentials;
import org.gcube.application.enm.service.conn.InformationSystemClient.FtpCredentials;
import org.gcube.application.enm.service.conn.InformationSystemClient.OpenModellerImage;
import org.gcube.application.enm.service.conn.InformationSystemClient.VmInstance;
import org.gcube.application.enm.service.conn.StorageClientUtil;
import org.gcube.contentmanagement.blobstorage.service.IClient;

import com.bsc.venusbes.venusclient.credentials.BESCredential;
import com.bsc.venusbes.venusclient.credentials.FTPCredential;
import com.bsc.venusbes.venusclient.credentials.StorageCredential;
import com.bsc.venusbes.venusclient.types.JobStatus;

/**
 * Extends the {@link ExecutionResource} for the COMPSs provider.
 * 
 * @author Erik Torres <ertorser@upv.es>
 */
public class COMPsExecutionResource extends ExecutionResource {	

	private static final String FTP_BASEDIR = "/openbiotest/enm";

	private final FtpCredentials ftpCred;
	private final BESCredential besCred;
	private final StorageCredential storageCred;
	private final COMPSsCredentials compssCred;
	private final OpenModellerImage omImage;

	public COMPsExecutionResource() {
		super();
		ftpCred = isClient.getCOMPSFtpCredentials();
		compssCred = isClient.getCOMPSsCredentials();
		besCred = new BESCredential(compssCred.getBesUsername(), 
				compssCred.getBesPassword());
		storageCred = new FTPCredential(compssCred.getStorageUsername(), 
				compssCred.getStoragePassword());
		omImage = isClient.getOpenModellerImage();
	}

	@Override
	public final String sumbitExperiment(final UUID uuid, 
			final ExperimentRequest params) {
		String jobId = null;
		boolean shouldClean = false;
		try {
			// Upload input files to the server
			uploadExperimentInput(uuid, params);
			shouldClean = true;
			// Submit the experiment
			final COMPSsClient enmService = new COMPSsClient(
					compssCred.getEndpoint());
			final VmInstance vmInstance = isClient.getVmInstance(
					params.getInstance());
			final String logsDir = absoluteFtpPath(getExperimentLogsDir(uuid));
			final String requestsDir = absoluteFtpPath(
					getExperimentRequestDir(uuid));
			final String testingPointsFile = absoluteFtpPath(
					getExperimentDir(uuid), params.getTestingPoints().getName());
			final String resultsDir = absoluteFtpPath(getExperimentResultsDir(uuid));
			logger.trace("Logs dir: '" + logsDir + "', Requests dir: '" 
					+ requestsDir + "', Testing points file: '" 
					+ testingPointsFile + ", Results dir: '" + resultsDir + "'");
			jobId = enmService.submitModel(besCred, storageCred, 
					omImage.getWallClock(), omImage.getVmImage(), 
					vmInstance.getDisSizeGB(), vmInstance.getCores(), 
					vmInstance.getMemoryGB(), vmInstance.getMinVMs(), 
					vmInstance.getMaxVMs(), omImage.getJobNamePrefix(), 
					omImage.getAppLocation(), logsDir, requestsDir, 
					testingPointsFile, resultsDir);
			if (jobId == null) {			
				throw new ExecutionException("Unknown error");
			}
		} catch (ExecutionException e) {
			if (shouldClean) {
				try {
					cleanExperiment(uuid);
				} catch (Exception ie) {
					// nothing to do
				}
			}
			jobId = null;
			logger.error("Failed to submit experiment: " 
					+ e.getLocalizedMessage());
		}
		return jobId;
	}

	private void uploadExperimentInput(final UUID uuid, 
			final ExperimentRequest params) throws ExecutionException {
		FtpClient ftp = null;
		boolean shouldClean = false;
		try {			
			ftp = connectFtp();
			// Create experiment directories
			final String[] dirs = { 
					getExperimentDir(uuid), 
					getExperimentRequestDir(uuid),
					getExperimentLogsDir(uuid),
					// getExperimentResultsDir(uuid)
			};
			for (String dirname : dirs) {
				ftp.makeDirectory(dirname);
			}
			shouldClean = true;	
			// Upload requests
			for (RequestType request : params.getRequests().getRequest()) {
				ftp.uploadString(request.getRequest(), 
						getExperimentRequestDir(uuid) + "/" + request.getName());
			}
			// Upload test points
			ftp.uploadString(params.getTestingPoints().getTestPoints(),
					getExperimentDir(uuid) + "/" + params.getTestingPoints()
					.getName());
		} catch (IOException e) {
			if (ftp != null && shouldClean) {
				try {
					ftp.deleteDir(getExperimentDir(uuid));
				} catch (IOException ioe) {
					// nothing to do
				}
			}
			throw new ExecutionException("Failed to upload experiment input: " 
					+ e.getLocalizedMessage());
		} finally {
			if (ftp != null) {				
				ftp.disconnect();
			}
		}
	}

	private FtpClient connectFtp() throws IOException {
		final FtpClient ftp = new FtpClient();
		ftp.connect(ftpCred.getServer(), ftpCred.getPort(), 
				ftpCred.getUsername(), ftpCred.getPassword());
		if (!ftp.changeDirectory(FTP_BASEDIR))
			throw new IOException("Cannot change the FTP working directory to: " 
					+ FTP_BASEDIR);
		return ftp;
	}

	private void cleanExperiment(final UUID uuid) throws ExecutionException {
		FtpClient ftp = null;
		try {			
			ftp = connectFtp();
			// Clean experiment directory
			ftp.deleteDir(getExperimentDir(uuid));
		} catch (IOException e) {			
			throw new ExecutionException("Failed to clean experiment directory: " 
					+ e.getLocalizedMessage());
		} finally {
			if (ftp != null) {				
				ftp.disconnect();
			}
		}
	}

	private String getExperimentDir(final UUID uuid) {
		return uuid.toString();
	}

	private String getExperimentRequestDir(final UUID uuid) {
		return getExperimentDir(uuid) + "/requests"; 
	}

	private String getExperimentLogsDir(final UUID uuid) {
		return getExperimentDir(uuid) + "/logs"; 
	}

	private String getExperimentResultsDir(final UUID uuid) {
		return getExperimentDir(uuid) + "/results"; 
	}

	private String baseFtpPath() {
		return "ftp://" + ftpCred.getServer() + FTP_BASEDIR + "/";
	}

	private String absoluteFtpPath(final String relativePath) {
		return absoluteFtpPath(relativePath, "");
	}

	private String absoluteFtpPath(final String relativePath, 
			final String filename) {
		return baseFtpPath() + relativePath + "/" + filename;
	}

	@Override
	public final ExperimentStatus getStatus(final UUID uuid, 
			final String jobId) {
		ExperimentStatus experimentStatus = null;
		final String[] jobIds = { jobId };
		final COMPSsClient enmService = new COMPSsClient(
				compssCred.getEndpoint());
		final List<JobStatus> status = enmService.OModellerStatus(besCred, 
				jobIds);		
		if (status != null && status.size() == 1 && status.get(0) != null) {
			experimentStatus = (new ObjectFactory()).createExperimentStatus();
			experimentStatus.setCompletenessPercentage(new BigDecimal(0));
			if (status.get(0).getStatus().equals("RUNNING_STAGE_OUT")) {
				experimentStatus.setStatus(StatusType.EXECUTING);
				experimentStatus.setCompletenessPercentage(new BigDecimal(100));
			} else if (status.get(0).getStatus().startsWith("RUNNING")) 
				experimentStatus.setStatus(StatusType.EXECUTING);
			else if (status.get(0).getStatus().equals("FINISHED")) {
				experimentStatus.setStatus(StatusType.FINISHED);
				experimentStatus.setCompletenessPercentage(new BigDecimal(100));
			}
			else if (status.get(0).getStatus().equals("FAILED")) 
				experimentStatus.setStatus(StatusType.FAILED);
			else if (status.get(0).getStatus().equals("CANCELLED")) 
				experimentStatus.setStatus(StatusType.CANCELLED);
			else experimentStatus.setStatus(StatusType.PENDING);
			experimentStatus.setDescription(status.get(0).getStatus());			
			try {
				if (status.get(0).getStatus().equals("RUNNING_EXECUTING"))
					experimentStatus.setCompletenessPercentage(new BigDecimal(
							enmService.getCompletness(besCred, jobIds)));
			} catch (NullPointerException e) {
				// nothing to do
			}
		}
		return experimentStatus;
	} 

	@Override
	public final ExperimentResults getResults(final UUID uuid, 
			final String jobId) {
		final ExperimentResults results = (new org.gcube.application.enm.common
				.xml.results.ObjectFactory()).createExperimentResults();
		FtpClient ftp = null;
		try {			
			ftp = connectFtp();
			// List results files
			final String[] resFiles = ftp.listFiles(getExperimentResultsDir(
					uuid));
			for (String resfile : resFiles) {
				final ResultType result = new ResultType();
				final String resLocation = absoluteFtpPath(
						getExperimentResultsDir(uuid), resfile);
				result.setRemoteJobId(jobId);
				result.setOutputLocation(resLocation);
				result.setFilename(new File(resfile).getName());
				results.getResult().add(result);
			}
		} catch (IOException e) {
			logger.error("Failed to get experiment results: "
					+ e.getLocalizedMessage());
		} finally {
			if (ftp != null) {				
				ftp.disconnect();
			}
		}
		return results;
	}

	@Override
	public final ExperimentLogs getLogs(final UUID uuid, final String jobId) {
		final ExperimentLogs logs = (new org.gcube.application.enm.common
				.xml.logs.ObjectFactory()).createExperimentLogs();
		FtpClient ftp = null;
		try {			
			ftp = connectFtp();
			// List log files
			final String[] logFiles = ftp.listFiles(getExperimentLogsDir(uuid));
			for (String logfile : logFiles) {
				final LogType log = new LogType();
				final String resLocation = absoluteFtpPath(
						getExperimentLogsDir(uuid), logfile);
				log.setRemoteJobId(jobId);
				log.setOutputLocation(resLocation);
				log.setFilename(new File(logfile).getName());
				logs.getLog().add(log);
			}
		} catch (IOException e) {
			logger.error("Failed to get experiment logs: "
					+ e.getLocalizedMessage());
		} finally {
			if (ftp != null) {				
				ftp.disconnect();
			}
		}
		return logs;
	}

	@Override
	public void cancelExperiment(final UUID uuid, 
			final String jobId) {
		final String[] jobIds = { jobId };
		final COMPSsClient enmService = new COMPSsClient(
				compssCred.getEndpoint());
		final List<String> messageList = enmService.terminateOModeller(besCred, 
				jobIds);
		if (messageList != null) {
			for (String msg : messageList) {
				if (msg != null)
					logger.trace(msg);
			}
			try {
				cleanExperiment(uuid);
			} catch (ExecutionException e) {
				// nothing to do
			}
		}
	}

	@Override
	public String getResultAsFile(final ResultType result, 
			final IClient storageClient, final String credentials) {
		String id = null;
		FtpClient ftp = null;
		try {
			final String remoteFilename = result.getOutputLocation().replace(
					baseFtpPath(), "");
			final String localFilename = result.getOutputLocation().replace(
					baseFtpPath(), StorageClientUtil.userBaseDir(credentials));
			ftp = connectFtp();
			final InputStream is = ftp.retrieveFileAsInputStream(remoteFilename);			
			id = storageClient.put(true).LFile(is).RFile(localFilename);
		} catch (IOException e) {
			logger.error("Failed to retrieve experiment result: "
					+ e.getLocalizedMessage());
		} catch (Exception ex) {
			logger.error("Unkonwn error found while retrieving experiment result: "
					+ ex.getLocalizedMessage());
		} finally {
			if (ftp != null) {
				ftp.disconnect();
			}
		}
		return id;
	}

	@Override
	public String getLogAsFile(final LogType log, final IClient storageClient
			, final String credentials) {
		String id = null;
		FtpClient ftp = null;
		try {
			final String remoteFilename = log.getOutputLocation().replace(
					baseFtpPath(), "");
			final String localFilename = log.getOutputLocation().replace(
					baseFtpPath(), StorageClientUtil.userBaseDir(credentials));
			ftp = connectFtp();
			final InputStream is = ftp.retrieveFileAsInputStream(remoteFilename);
			id = storageClient.put(true).LFile(is).RFile(localFilename);
		} catch (IOException e) {
			logger.error("Failed to retrieve experiment log: "
					+ e.getLocalizedMessage());
		} catch (Exception ex) {
			logger.error("Unkonwn error found while retrieving experiment log: "
					+ ex.getLocalizedMessage());
		} finally {
			if (ftp != null) {
				ftp.disconnect();
			}
		}
		return id;
	}

	@Override
	public void cleanExperimentOuput(final UUID uuid, final String jobId) {		
		try {
			cleanExperiment(uuid);
		} catch (ExecutionException e) {
			// nothing to do
		}		
	}

}
