package org.gcube.application.enm.service.plugins.comps;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import org.gcube.application.enm.common.xml.logs.ExperimentLogs;
import org.gcube.application.enm.common.xml.logs.LogType;
import org.gcube.application.enm.common.xml.request.ExperimentRequest;
import org.gcube.application.enm.common.xml.results.ExperimentResults;
import org.gcube.application.enm.common.xml.results.ResultType;
import org.gcube.application.enm.common.xml.status.ExperimentStatus;
import org.gcube.application.enm.common.xml.status.StatusType;
import org.gcube.application.enm.service.ExecutionResource;
import org.gcube.application.enm.service.ExperimentUtils;
import org.gcube.application.enm.service.GenericJob;
import org.gcube.application.enm.service.concurrent.JobMonitor;
import org.gcube.application.enm.service.conn.JobUpdate;
import org.gcube.application.enm.service.conn.StorageClientUtil;
import org.gcube.application.enm.service.util.XmlHelper;
import org.gcube.contentmanagement.blobstorage.service.IClient;

/**
 * Extends the {@link GenericJob} for the COMPSs provider.
 * 
 * @author Erik Torres <ertorser@upv.es>
 */
public class COMPSsJob extends GenericJob {

	private static final int INITIALIZATION = -1;
	private static final int SUPERVISION = 0;
	private static final int FINALIZATION = 1;

	public COMPSsJob(final UUID uuid, final ExperimentRequest request) {
		super(uuid, request);
	}

	public COMPSsJob(final UUID uuid, final ExperimentRequest request, 
			final ExperimentStatus status, final ExperimentResults results, 
			final ExperimentLogs logs) {
		super(uuid, request, status, results, logs);
	}

	@Override
	public ExecutionResource getExecutionResource() {
		return new COMPsExecutionResource();
	}

	@Override
	public String call() throws Exception {
		// Create an execution resource for this experiment
		final ExecutionResource execRes = getExecutionResource();
		// Check whether the job is cancelled
		if (isCanceled()) {
			for (final String jobIdItem : currentJobIds()) {
				execRes.cancelExperiment(uuid, jobIdItem);
				logger.trace("Job '" + uuid + "' with local id '" + jobIdItem 
						+ "' was cancelled");
			}
			// update the status
			status.setStatus(StatusType.CANCELLED);
			status.setDescription("The experiment was cancelled by the user");
			status.setCompletenessPercentage(new BigDecimal(0));			
			status.setEndDate(XmlHelper.now());
			JobMonitor.get().updateJob(this, new JobUpdate().status(true)
					.results(false).logs(false));
			logger.trace("Job '" + uuid + "' was cancelled");
		} else {
			// Select and execute an action from the execution trace
			switch (currentStep()) {
			case INITIALIZATION: // Submit the experiment			
				final String jobId = execRes.sumbitExperiment(uuid, request);
				next(SUPERVISION, jobId);
				JobMonitor.get().schedulePendingJob(this);
				logger.trace("Job '" + uuid.toString() + "' was successfully " 
						+ "submitted with local id '" + jobId + "'");
				break;
			case FINALIZATION:
				// Update the results and the logs from the remote storage				
				for (final String jobIdItem : currentJobIds()) {
					// Results
					final ExperimentResults jobResults = execRes.getResults(
							uuid, jobIdItem);
					for (final ResultType result : jobResults.getResult()) {
						results.getResult().add(result);
					}
					// Logs
					final ExperimentLogs jobLogs = execRes.getLogs(
							uuid, jobIdItem);
					for (final LogType log : jobLogs.getLog()) {
						logs.getLog().add(log);
					}
				}
				// Copy the results and the logs to the local storage
				final IClient storageClient = StorageClientUtil.client();
				for (final ResultType result : results.getResult()) {
					if (result.isRemote()) {
						final String id = execRes.getResultAsFile(result, 
								storageClient, request.getCredentials());
						result.setOutputLocation(id);
						result.setRemote(false);
					}
				}
				for (final LogType log : logs.getLog()) {
					if (log.isRemote()) {
						final String id = execRes.getLogAsFile(log, 
								storageClient, request.getCredentials());
						log.setOutputLocation(id);
						log.setRemote(false);
					}
				}
				// Clean the remote storage
				for (final String jobIdItem : currentJobIds()) {
					execRes.cleanExperimentOuput(uuid, jobIdItem);
				}
				status.setEndDate(XmlHelper.now());
				JobMonitor.get().updateJob(this, new JobUpdate().status(true)
						.results(true).logs(true));
				logger.trace("Job '" + uuid.toString() + "' was cleared");
				break;
			case SUPERVISION:
			default:
				boolean hasFinished = true;
				int completenessPercentage = 0;
				String desc = "";
				if (currentJobIds().size() > 0) {
					final List<StatusType> statusList = 
							new ArrayList<StatusType>();
					for (final String jobIdItem : currentJobIds()) {
						final ExperimentStatus jobStatus = execRes.getStatus(
								uuid, jobIdItem);
						if (jobStatus != null) {
							hasFinished = hasFinished 
									&& ExperimentUtils.hasFinished(jobStatus);
							completenessPercentage += jobStatus
									.getCompletenessPercentage().intValue();
							desc += jobIdItem + ":" + jobStatus.getDescription() 
									+ " ";
							statusList.add(jobStatus.getStatus());
							logger.trace("Job '" + uuid + "' with local id '" 
									+ jobIdItem + "' status: " 
									+ jobStatus.getStatus()
									+ ", % of completeness: " 
									+ jobStatus.getCompletenessPercentage()
									+ ", remote status: " 
									+ jobStatus.getDescription());
						}
					}
					completenessPercentage /= currentJobIds().size();
					// update the status
					status.setStatus(homogeneousStatus(statusList));
					status.setCompletenessPercentage(new BigDecimal(
							completenessPercentage));
					status.setDescription(desc);
					JobMonitor.get().updateJob(this, new JobUpdate().status(true)
							.results(false).logs(false));
				} else hasFinished = false;
				if (hasFinished)
					next(FINALIZATION, currentJobIds());
				JobMonitor.get().schedulePendingJob(this);
				break;
			}
		}
		return uuid.toString();
	}

	public StatusType homogeneousStatus(final List<StatusType> statusList) {
		StatusType status = StatusType.PENDING;
		int total = 0, executing = 0, finished = 0;
		for (StatusType item : statusList) {
			total++;
			if (item.equals(StatusType.FAILED) 
					|| item.equals(StatusType.CANCELLED))
				return item;
			else if (item.equals(StatusType.EXECUTING))
				executing++;
			else if (item.equals(StatusType.FINISHED))
				finished++;
		}
		if (total == executing) 
			status = StatusType.EXECUTING;
		else if (total == finished)
			status = StatusType.FINISHED;
		return status;
	}	

}
