

package org.gcube.accounting.datamodel;

import java.io.Serializable;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

import org.gcube.accounting.datamodel.serialization.MapAdaptor;
import org.gcube.accounting.exception.InvalidValueException;

/**
 * A base implementation of the UsageRecord interface
 */

@XmlRootElement(name = "usageRecord")
@XmlAccessorType(XmlAccessType.PROPERTY)
public class RawUsageRecord implements UsageRecord, Serializable {

	/**
	 * 
	 */
	private static final long serialVersionUID = 5714960796256372093L;

	/**
	 * The id of the usage record
	 */
	private String id;

	/**
	 * When the record was created
	 */
	private Calendar createTime;

	/**
	 * When the record was last modified
	 */
	private Calendar modifyTime;

	/**
	 * Who created the record; usually differs from the consumer. It can be the
	 * connector, the middleware, etc..
	 */
	private String creatorId;

	/**
	 * The id of the entity who consumed the resource
	 */
	private String consumerId;

	/**
	 * Fully-qualifield consumer identity. Includes the full path of the
	 * consumer in the organisation.
	 */
	private String fullyQualifiedConsumerId;

	private Calendar startTime;
	private Calendar endTime;

	/**
	 * The owner of the resource
	 */
	private String resourceOwner;
	
	/**
	 * The scope of the resource
	 */
	private String resourceScope;

	/**
	 * resource-specific properties. These most likely have dedicated
	 * getters/setters in subclasses
	 */
	//@XmlJavaTypeAdapter(MapAdaptor.class)
	private Map<String, String> extension;

	/**
	 * Custom properties (user-defined)
	 */
//	@XmlJavaTypeAdapter(MapAdaptor.class)
//	@XmlElement(name = "customProperties")
	private Map<String, String> custom;

	/**
	 * If replaced by an aggregated UR, this field is a link to it
	 */
	private String aggregatedId;

	/**
	 * The type of the resource this UR refers to
	 */
	private String resourceType;

	/**
	 * Base constructor. Just initialize the id
	 */
	public RawUsageRecord() {
		this.id = UUID.randomUUID().toString();
		this.extension = new HashMap<String, String>();
		this.custom = new HashMap<String, String>();
	}

	/**
	 * Instantiate this record using values from the given record
	 * 
	 * @param r
	 */
	public RawUsageRecord(UsageRecord r) throws InvalidValueException {
		this();
		this.setAggregatedId(r.getAggregatedId());
		this.setConsumerId(r.getConsumerId());
		this.setFullyQualifiedConsumerId(r.getFullyQualifiedConsumerId());
		this.setCreateTime(r.getCreateTime());
		this.setModifyTime(r.getModifyTime());
		this.setCreatorId(r.getCreatorId());
		this.setEndTime(r.getEndTime());
		this.setId(r.getId());
		this.setResourceScope(r.getResourceScope());
		this.setResourceOwner(r.getResourceOwner());
		this.setResourceType(r.getResourceType());
		this.setStartTime(r.getStartTime());
		for (String s : r.getResourceSpecificProperties().keySet()) {
			this.setResourceSpecificProperty(s, r
					.getResourceSpecificProperty(s));
		}
		for (String s : r.getCustomProperties().keySet()) {
			this.setCustomProperty(s, r.getCustomProperty(s));
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public Calendar getCreateTime() {
		return createTime;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setCreateTime(Calendar createTime) {
		this.createTime = createTime;
	}

	/**
	 * {@inheritDoc}
	 */
	public Calendar getModifyTime() {
		return modifyTime;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setModifyTime(Calendar modifyTime) {
		this.modifyTime = modifyTime;
	}

	/**
	 * {@inheritDoc}
	 */
	@XmlElement(name = "creatorId")
	public String getCreatorId() {
		return creatorId;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setCreatorId(String creatorId) {
		this.creatorId = creatorId;
	}

	/**
	 * {@inheritDoc}
	 */
	@XmlElement(name = "consumerId")
	public String getConsumerId() {
		return consumerId;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setConsumerId(String consumerId) {
		this.consumerId = consumerId;
	}

	public String getFullyQualifiedConsumerId() {
		return this.fullyQualifiedConsumerId;
	}
	
	public void setFullyQualifiedConsumerId(String fqcid) {
		this.fullyQualifiedConsumerId = fqcid;
	}

	/**
	 * @return the startTime
	 */
	@XmlElement(name = "startTime")
	public Calendar getStartTime() {
		return startTime;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setStartTime(Calendar startTime) throws InvalidValueException {
		if(this.getEndTime()!=null && startTime.after(this.getEndTime()))
			throw new InvalidValueException("startTime cannot be after endTime");
		this.startTime = startTime;
	}


	/**
	 * {@inheritDoc}
	 */
	public void setEndTime(Calendar endTime) throws InvalidValueException {
		if(this.getStartTime()!=null && endTime.before(this.getStartTime()))
			throw new InvalidValueException("endTime cannot be before startTime");
		this.endTime = endTime;
	}

	/**
	 * {@inheritDoc}
	 */
	@XmlElement(name = "endTime")
	public Calendar getEndTime() {
		return endTime;
	}

	/**
	 * {@inheritDoc}
	 */
	@XmlElement(name = "id")
	public String getId() {
		return id;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setResourceSpecificProperty(String name, String value) {
		this.extension.put(name, value);
	}

	/**
	 * {@inheritDoc}
	 */
	public String getResourceSpecificProperty(String key) {
		return this.extension.get(key);
	}

	/**
	 * {@inheritDoc}
	 */
	public void setCustomProperty(String name, String value) {
		this.custom.put(name, value);
	}

	/**
	 * {@inheritDoc}
	 */
	public String getCustomProperty(String name) {
		return this.custom.get(name);
	}

	/**
	 * {@inheritDoc}
	 */
	public void setId(String id) {
		this.id = id;
	}

	/**
	 * {@inheritDoc}
	 */
	public String getAggregatedId() {
		return aggregatedId;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setAggregatedId(String aggregatedId) {
		this.aggregatedId = aggregatedId;
	}

	/**
	 * {@inheritDoc}
	 */
	@XmlElement(name = "resourceType")
	public String getResourceType() {
		return resourceType;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setResourceType(String resourceType) {
		this.resourceType = resourceType;
	}

	/**
	 * {@inheritDoc}
	 */
	@XmlElement(name = "resourceOwner")
	public String getResourceOwner() {
		return this.resourceOwner;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setResourceOwner(String owner) {
		this.resourceOwner = owner;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setResourceScope(String scope) {
		this.resourceScope = scope;
		
	}

	/**
	 * {@inheritDoc}
	 */
	@XmlElement(name = "resourceScope")
	public String getResourceScope() {
		return this.resourceScope;
	}

	/**
	 * {@inheritDoc}
	 */
	@XmlJavaTypeAdapter(MapAdaptor.class)
	@XmlElement(name = "resourceSpecificProperties", required = true)

	public Map<String, String> getResourceSpecificProperties() {
		return this.extension;
	}

	/**
	 * {@inheritDoc}
	 */
	@XmlJavaTypeAdapter(MapAdaptor.class)
	public Map<String, String> getCustomProperties() {
		return this.custom;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setCustomProperties(Map<String, String> props) {
		this.custom = props;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setResourceSpecificProperties(Map<String, String> props) {
		this.extension = props;
	}
	
	public void validate() {
		
	}



}
