package org.gcube.accounting.usagetracker.persistence;

import java.util.Calendar;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.TreeSet;

import org.apache.log4j.Logger;

import com.mongodb.DBObject;

class Record {
	
	private static Logger logger = Logger.getLogger(Record.class);

	
	private String resourceOwner;
	private String consumerId;
	private String dataType;
	private String scope;
	private Long timestamp;
	private Double value;
	
	
	public Record(DBObject obj, String valueKey) throws Exception{
		DBObject idO = (DBObject) obj.get("_id");
		this.resourceOwner = idO.get("resourceOwner").toString();
		this.consumerId = idO.get("consumerId").toString();
		this.dataType = idO.get("rsp:dataType").toString();
		this.scope = idO.get("resourceScope").toString();
		this.value = Double.valueOf(obj.get(valueKey).toString());
		
		if(this.value < 0){
			throw new Exception("Negative value found ("+this.value+"). Discarding");
		}
		
		Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
		cal.set(Calendar.HOUR_OF_DAY, 0);
		cal.clear(Calendar.MINUTE);
		cal.clear(Calendar.SECOND);
		cal.clear(Calendar.MILLISECOND);
		cal.set(Calendar.DAY_OF_MONTH, Integer.valueOf(idO.get("day").toString()));
		cal.set(Calendar.MONTH, Integer.valueOf(idO.get("month").toString()) - 1);
		cal.set(Calendar.YEAR, Integer.valueOf(idO.get("year").toString()));
		
		this.timestamp = cal.getTimeInMillis();
		
		logger.debug("Record created with value " + this.getValue() + " and timestamp " + this.getTimestamp());
		
		
	}
	
	public Record(String resourceOwner, String consumerId, String resourceType,
			String scope, Long timestamp, Double value) {
		super();
		this.resourceOwner = resourceOwner;
		this.consumerId = consumerId;
		this.dataType = resourceType;
		this.scope = scope;
		this.timestamp = timestamp;
		this.value = value;
	}

	public Long getTimestamp() {
		return this.timestamp;
	}

	public String getResourceOwner() {
		return resourceOwner;
	}

	public String getConsumerId() {
		return consumerId;
	}

	public String getResourceType() {
		return dataType;
	}

	public String getScope() {
		return scope;
	}
	
	public Double getValue() {
		return value;
	}
}

class Helper {
	public static String getKeyExcluding(Record r, String excluded) {
		String out = "";
		if(!"resourceOwner".equals(excluded))
			out+=r.getResourceOwner();
		if(!"consumerId".equals(excluded))
			out+=r.getConsumerId();
		if(!"rsp:dataType".equals(excluded))
			out+=r.getResourceType();
		if(!"resourceScope".equals(excluded))
			out+=r.getScope();
		return out;
	}
	
	public static String getDimensionValue(Record r, String key) throws Exception {
		if("resourceOwner".equals(key))
			return r.getResourceOwner();
		if("consumerId".equals(key))
			return r.getConsumerId();
		if("rsp:dataType".equals(key))
			return r.getResourceType();
		if("resourceScope".equals(key))
			return r.getScope();
		return "all";
//		throw new Exception("unknown dimension " + key);
	}
}

class RecordTimeComparator implements Comparator<Record> {
	public int compare(Record r1, Record r2) {
		return r1.getTimestamp().compareTo(r2.getTimestamp());
	}
}

class SortedRecords {
	Set<Record> records = new TreeSet<Record>(new RecordTimeComparator());
	public void add(Record r) {
		this.records.add(r);
	}
	public Record getEarlierRecord(Long timestamp) {
		Record previous = null;
		for(Record r:records) {
			if(r.getTimestamp()>timestamp)
				return previous;
			previous = r;
		}
		return previous;
	}
}

class DimensionValueMap {
	private String dimensionKey;
//	private String dimensionValue;
	private Map<String, SortedRecords> fields2records;
	
	public DimensionValueMap(String dimensionKey) {
		this.dimensionKey = dimensionKey;
		this.fields2records = new HashMap<String, SortedRecords>();
	}
	
	
	public void add(Record r) {
		String key = Helper.getKeyExcluding(r, this.dimensionKey);
		if(!this.fields2records.containsKey(key))
			this.fields2records.put(key, new SortedRecords());
		this.fields2records.get(key).add(r);
	}
	
	public Double getSumAt(Long timestamp) {
		Double out = 0d;
//		System.out.println(fields2records.keySet().size());
		for(String key:fields2records.keySet()) {
			Record r = fields2records.get(key).getEarlierRecord(timestamp);
			if(r!=null) {
				out+=r.getValue();
//				System.out.println(r.getValue());
			}
		}
		return out;
	}
	
}

class AggregatedResult {
	Long timestamp;
	String dimension;
	Double value;
	
	public AggregatedResult(Long timestamp, String dimension, Double value) {
		super();
		this.timestamp = timestamp;
		this.dimension = dimension;
		this.value = value;
	}
	public Long getTimestamp() {
		return timestamp;
	}
	public String getDimension() {
		return dimension;
	}
	public Double getValue() {
		return value;
	}
}



public class ResultSet {
	
	public static final Double MILLS_IN_DAY = 60*60*24*1000d;
	
	private String dimensionKey;
	
	private Map<String, DimensionValueMap> dimensionValue2records;
	
	public ResultSet(String dimensionKey) {
		this.dimensionKey = dimensionKey;
		this.dimensionValue2records = new HashMap<>();
	}

	private DimensionValueMap getValueMap(String dimensionValue) {
		if(!dimensionValue2records.containsKey(dimensionValue))
			dimensionValue2records.put(dimensionValue, new DimensionValueMap(dimensionValue));
		return dimensionValue2records.get(dimensionValue);
	}
	
	public void add(Record r) throws Exception {
		String dimensionValue = Helper.getDimensionValue(r, this.dimensionKey);
//		System.out.println(dimensionValue);
//		System.out.println(this.getValueMap(dimensionValue));
		this.getValueMap(dimensionValue).add(r);
	}
	
	public Double getSumAt(String dimensionValue, Long timestamp) {
		return this.getValueMap(dimensionValue).getSumAt(timestamp);
	}
	
	public AggregatedResult getAggregatedResult(String dimensionValue, Long timestamp) {
		return new AggregatedResult(timestamp, dimensionValue, this.getValueMap(dimensionValue).getSumAt(timestamp));
	}

	public Double getAverageInRange(String dimensionValue, Long from, Long to) {
		Double out = 0d;
		List<Long> intervals = UsageTrackerDB.getIntervals(from, to, "day");
		
		Long previous = intervals.get(0);
		for(int i=1; i< intervals.size(); i++) {
			Long dayEnd = intervals.get(i);
			
			
			Double value = this.getValueMap(dimensionValue).getSumAt(dayEnd);
			value = value * (dayEnd - previous) / MILLS_IN_DAY; 
			
			out+=value;
			
			previous = dayEnd;
		}
		return out/(intervals.size() - 1);
	}
	
	public Double getVarInRange(String dimensionValue, Long from, Long to) {
		Double out = 0d;
		Double avg = this.getAverageInRange(dimensionValue, from, to);
		for(Long current=from+1; current<=to; current++) {
			Double value = this.getValueMap(dimensionValue).getSumAt(current);
			out += Math.pow(value-avg, 2);
		}
		return out/(to-from);
	}

	public Double getMinInRange(String dimensionValue, Long from, Long to) {
		Double out = Double.MAX_VALUE;
		for(Long current=from+1; current<=to; current++) {
			Double value = this.getValueMap(dimensionValue).getSumAt(current);
			out = Math.min(out, value);
		}
		return out;
	}

	public Double getMaxInRange(String dimensionValue, Long from, Long to) {
		Double out = 0d;
		for(Long current=from+1; current<=to; current++) {
			Double value = this.getValueMap(dimensionValue).getSumAt(current);
			out = Math.max(out, value);
		}
		return out;
	}

	
	public Set<String> getDimensionValues() {
		return this.dimensionValue2records.keySet();
	}
	
	
	public static void main(String[] args) throws Exception {
		
		String[] people = {
				"nunzio",
				"ciro",
				"gabriele",
				"philip",
				"davide",
				"cosimo",
				"giacinta",
				"maria",
				"rita",
				"mario",
				"tiziana",
				"ettore",
				"achille",
				"giuseppe",
				"loretta",
				"piero",
				"matteo",
				"giacomo",
				"giovanni",
				"giocondo",
				"antonio",
				"antonietta",
				"luca",
				"mara",
				"martina",
				"anna",
				"andrea",
				"lino",
				"pasquale",
				"lucio",
				"massi",
				"gianpaolo",
				"jurgen",
				"jessica",
				"rosella"
		};
		
		String[] scopes = {
				"gcube",
				"devsec",
				"testing",
				"fishery",
				"eo"
		};
		
		String[] types = {
				"blob",
				"object",
				"database",
				"tree",
				"bigtable"
		};
		
		Long mindate = 0l;
		Long maxdate = 100l;
		
		Long maxValue = 10000l;
		
		ResultSet rm = new ResultSet("resourceOwner");

		for(int i=0; i<10000; i++) {
			String consumerId = people[(int)(Math.random()*people.length)];
			String resourceOwner = people[(int)(Math.random()*people.length)];
			String type = types[(int)(Math.random()*types.length)];
			String scope = scopes[(int)(Math.random()*scopes.length)];
			Long timestamp = mindate+((long)(Math.random()*(maxdate-mindate)));
			Double value = (Math.random()*maxValue);
			System.out.println(consumerId+" "+resourceOwner+" "+type+" "+scope+" "+timestamp+" "+value);
			rm.add(new Record(resourceOwner, resourceOwner, type, scope, timestamp, value));
		}
		
//		rm.add(new Record("gianpaolo.coro", "paolo.fabriani", "tree", "gcube", 10l, 500l));
//		rm.add(new Record("gianpaolo.coro", "paolo.fabriani", "db", "gcube", 20l, 200l));
//		rm.add(new Record("gianpaolo.coro", "paolo.fabriani", "blobs", "devsec", 25l, 100l));
//		rm.add(new Record("gianpaolo.coro", "paolo.fabriani", "tree", "gcube", 30l, 100l));
//		rm.add(new Record("loredana", "loredana", "tree", "gcube", 15l, 5l));
		
		for(String dimension:rm.getDimensionValues()) {
			System.out.println("***** " + dimension + " *****");
			AggregatedResult prev = null;
			for(long i=mindate; i<maxdate; i++) {
				AggregatedResult r = rm.getAggregatedResult(dimension, 0l+i);
				if(prev==null || !prev.getValue().equals(r.getValue())) {
					System.out.println(i+". "+r.getValue());
				}
				long mod = 10;
				if(i%mod==0) {
					long previousTimestamp = Math.max(i-mod, -1);
//					System.out.println(previousTimestamp + " "+ i);
					System.out.println("Min/Avg/Max/Var: "
							+ "("+previousTimestamp+"-"+i+") "
							+ rm.getMinInRange(dimension, previousTimestamp, i)
							+ " / " + rm.getAverageInRange(dimension, previousTimestamp, i)
							+ " / " + rm.getMaxInRange(dimension, previousTimestamp, i)
							+ " / " + rm.getVarInRange(dimension, previousTimestamp, i)
						);
				}
				prev = r;
			}
		}
		
	}

}
