

package org.gcube.accounting.usagetracker.persistence;

import java.net.UnknownHostException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.TreeSet;
import java.util.Vector;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;
import org.gcube.accounting.common.helper.ISClientConnector;
import org.gcube.accounting.datamodel.RawUsageRecord;
import org.gcube.accounting.datamodel.UsageRecord;
import org.gcube.accounting.datamodel.adapter.EntryFilterRating;
import org.gcube.accounting.datamodel.aggregation.AggregationField;
import org.gcube.accounting.datamodel.aggregation.ResultOptions;
import org.gcube.accounting.datamodel.query.QueryClause;
import org.gcube.accounting.datamodel.query.QueryOperator;
import org.gcube.accounting.datamodel.reports.Report;
import org.gcube.accounting.datamodel.reports.aggregation.AggregatedResult;
import org.gcube.accounting.datamodel.reports.aggregation.FilterRating;
import org.gcube.accounting.datamodel.reports.aggregation.Metric;
import org.gcube.accounting.exception.InvalidValueException;
import org.gcube.accounting.exception.NotFoundException;
import org.gcube.accounting.usagetracker.configuration.Configuration;
import org.joda.time.DateTime;

import com.mongodb.AggregationOutput;
import com.mongodb.BasicDBList;
import com.mongodb.BasicDBObject;
import com.mongodb.DB;
import com.mongodb.DBCollection;
import com.mongodb.DBCursor;
import com.mongodb.DBObject;
import com.mongodb.MongoClient;
import com.mongodb.WriteConcern;


public class UsageTrackerDB {

	// a db-specific prefix for resource-specific properties
	private static final String RSP_PREFIX = "rsp:";

	// a db-specific prefix for custom properties
	//private static final String CP_PREFIX = "cp:";

	// encoding for special characters

	private static final String DOLLAR = "\\\\DOL";
	private static final String DOT = "\\\\DOT";
	private static final String BACKSLASH = "\\\\\\\\";

	/**
	 * The singleton instance
	 */
	private static UsageTrackerDB utdb;
	private MongoClient mongo;
	private DB db;

	/**
	 * The list of mandatory fields 
	 */
	private static DBField[] baseFields = {
		DBField.RESOURCE_TYPE,
		DBField.CONSUMER_ID,
		DBField.FULLY_QUALIFIED_CONSUMER_ID,
		DBField.START_TIME, 
		DBField.END_TIME, 
		DBField.CREATOR_ID, 
		DBField.UR_ID,
		DBField.CREATE_TIME,
		DBField.MODIFY_TIME,
		DBField.RESOURCE_OWNER,
		DBField.RESOURCE_SCOPE
	};

	private static Logger logger = Logger.getLogger(UsageTrackerDB.class);

	/**
	 * Constructor is private to implement singleton
	 * @param hostname
	 * @param dbname
	 * @param port
	 * @throws UnknownHostException
	 */
	private UsageTrackerDB(String hostname, String dbname, int port)
			throws UnknownHostException {

		this.mongo = new MongoClient(hostname, port);
		this.db = mongo.getDB(dbname);

		//ensure indexes for all collections
		this.ensureIndexesForUsageRecords(this.getUsageRecordsCollection());
	}

	/**
	 * Constructor is private to implement singleton
	 * @param hostname
	 * @param dbname
	 * @param port
	 * @param username
	 * @param password
	 * @throws UnknownHostException
	 */
	private UsageTrackerDB(String hostname, String dbname, int port, String username, String password)
			throws UnknownHostException {
		this(hostname, dbname, port);
		this.db.authenticate(username, password.toCharArray());
	}

	/**
	 * Check whether the field 'field' is a basic field
	 * 
	 * @param f
	 * @return
	 */
	private boolean isBaseField(String field) {
		for (DBField f : baseFields)
			if (f.toString().equals(field))
				return true;
		return false;
	}

	/**
	 * Singleton
	 * 
	 * @return
	 * @throws UnknownHostException
	 */
	public static UsageTrackerDB getStorage() throws UnknownHostException {
		if (utdb == null) {
			Configuration conf = Configuration.getInstance();
			if(conf.isRemoteDBEnabled()){
				ISClientConnector gCubeISClient = new ISClientConnector();
				String[] endpoints = gCubeISClient.getServerRRFws();
				//missing logic
			}
			else
				if(conf.isDBAuthnEnabled()){
					utdb = new UsageTrackerDB(conf.getDBHost(), conf.getDBName(), conf.getDBPort(), 
							conf.getDBUser(),conf.getDBPassword());
				}
				else
					utdb = new UsageTrackerDB(conf.getDBHost(), conf.getDBName(), conf.getDBPort());
		}
		return utdb;
	}

	/**
	 * Create a storage object starting from the UR
	 * @param ur
	 * @return
	 */
	private DBObject createDBObject(UsageRecord ur) {
		BasicDBObject doc = new BasicDBObject();
		doc.put(DBField.UR_ID.toString(), ur.getId());
		if(ur.getCreateTime()!=null)
			doc.put(DBField.CREATE_TIME.toString(), ur.getCreateTime());

		doc.put(DBField.CREATOR_ID.toString(), ur.getCreatorId());
		doc.put(DBField.CONSUMER_ID.toString(), ur.getConsumerId());
		doc.put(DBField.FULLY_QUALIFIED_CONSUMER_ID.toString(), ur.getFullyQualifiedConsumerId());
		doc.put(DBField.RESOURCE_TYPE.toString(), ur.getResourceType());
		doc.put(DBField.RESOURCE_OWNER.toString(), ur.getResourceOwner());
		doc.put(DBField.RESOURCE_SCOPE.toString(), ur.getResourceScope());

		if (ur.getStartTime() != null){
			logger.debug("createDBObject.startTime = "+ur.getStartTime());

			doc.put(DBField.START_TIME.toString(),ur.getStartTime());
		}
		if (ur.getEndTime() != null){

			doc.put(DBField.END_TIME.toString(),ur.getEndTime());
		}
		Map<String, String> properties = ur.getResourceSpecificProperties();
		for (String key : properties.keySet()) {
			if (!doc.keySet().contains(key)) {
				try {
					Long value = Long.parseLong(properties.get(key));
					doc.put(RSP_PREFIX+encodeKey(key), value);
				}
				catch(Exception e) {
					try {
						Double value = Double.parseDouble(properties.get(key));
						doc.put(RSP_PREFIX+encodeKey(key), value);
					}
					catch(Exception e2) {
						doc.put(RSP_PREFIX+encodeKey(key), properties.get(key));
					}
					doc.put(RSP_PREFIX+encodeKey(key), properties.get(key));
				}
			}
		}
		//		properties = ur.getCustomProperties();
		//		for (String key : properties.keySet()) {
		//			if (!doc.keySet().contains(key)) {
		//				try {
		//					Long value = Long.parseLong(properties.get(key));
		//					doc.put(CP_PREFIX+encodeKey(key), value);
		//				}
		//				catch(Exception e) {
		//					try {
		//						Double value = Double.parseDouble(properties.get(key));
		//						doc.put(CP_PREFIX+encodeKey(key), value);
		//					}
		//					catch(Exception e2) {
		//						doc.put(CP_PREFIX+encodeKey(key), properties.get(key));
		//					}
		//					doc.put(CP_PREFIX+encodeKey(key), properties.get(key));
		//				}
		//			}
		//		}
		return doc;
	}

	/**
	 * Create an UsageRecord object starting from the storage object
	 * @param dbo
	 * @return
	 */
	private RawUsageRecord createUsageRecord(DBObject dbo) {
		RawUsageRecord out = new RawUsageRecord();
		out.setId((String) dbo.get(DBField.UR_ID.toString()));
		out.setConsumerId((String) dbo.get(DBField.CONSUMER_ID.toString()));
		out.setFullyQualifiedConsumerId((String)dbo.get(DBField.FULLY_QUALIFIED_CONSUMER_ID.toString()));
		out.setCreatorId((String) dbo.get(DBField.CREATOR_ID.toString()));
		out.setResourceType((String) dbo.get(DBField.RESOURCE_TYPE.toString()));
		out.setResourceOwner((String) dbo.get(DBField.RESOURCE_OWNER.toString()));
		out.setResourceScope((String) dbo.get(DBField.RESOURCE_SCOPE.toString()));
		if (dbo.get(DBField.START_TIME.toString()) != null) {
			//Calendar c = Calendar.getInstance();
			//c.setTimeInMillis((Long)dbo.get(DBField.START_TIME.toString()));
			try {
				out.setStartTime((Date)dbo.get(DBField.START_TIME.toString()));
			}
			catch(InvalidValueException e) {
				logger.error(e.getMessage(), e);
			}
		}
		if (dbo.get(DBField.END_TIME.toString()) != null) {
			//Calendar c = Calendar.getInstance();
			//c.setTimeInMillis((Long)dbo.get(DBField.END_TIME.toString()));
			try {
				out.setEndTime((Date)dbo.get(DBField.END_TIME.toString()));
			}
			catch(InvalidValueException e) {
				logger.error(e.getMessage(), e);
			}
		}
		if (dbo.get(DBField.CREATE_TIME.toString()) != null) {
			//Calendar c = Calendar.getInstance();
			//c.setTimeInMillis((Long)dbo.get(DBField.CREATE_TIME.toString()));
			//out.setCreateTime(c);
		}
		//		if (dbo.get(DBField.MODIFY_TIME.toString()) != null) {
		//			Calendar c = Calendar.getInstance();
		//			c.setTimeInMillis((Long)dbo.get(DBField.MODIFY_TIME.toString()));
		//			out.setModifyTime(c);
		//		}
		for (String key : dbo.keySet()) {
			if (!this.isBaseField(key)) {
				Object value = dbo.get(key);
				if (value instanceof String || value instanceof Double || value instanceof Long) {
					if(key.startsWith(RSP_PREFIX)) {
						//logger.debug("r-s property:<" + key.substring(RSP_PREFIX.length()) + ":" + dbo.get(key).toString()+">");
						//out.setResourceSpecificProperty(decodeKey(key.substring(RSP_PREFIX.length())), dbo.get(key).toString());
						out.setResourceSpecificProperty(decodeKey(key), dbo.get(key).toString());

					}
					//					if(key.startsWith(CP_PREFIX)) {
					//						logger.debug("custom property:<" + key.substring(CP_PREFIX.length()) + ":" + dbo.get(key).toString()+">");
					//						out.setCustomProperty(decodeKey(key.substring(CP_PREFIX.length())), dbo.get(key).toString());
					//					}
				}
			}
		}
		return out;
	}



	/**
	 * Add the given usage record
	 * @param ur
	 * @throws JMSException 
	 */
	public void addUsageRecord(UsageRecord ur) {
		// set the create time to 'now'
		ur.setCreateTime(new Date());
		// set also the modify time to 'now'
		//		try {
		//			ur.setModifyTime(ur.getCreateTime());
		//		}
		//		catch(InvalidValueException e) {
		//			// this should never happen
		//			logger.warn(e.getMessage(), e);
		//		}
		// retrieve collection

		DBCollection coll = getUsageRecordsCollection();
		// insert object
		logger.info(String.format("inserting new record %s", ur.getId()));
		coll.insert(this.createDBObject(ur));

		// update all the support collections (e.g. resource types, resource scopes, ...)
		logger.info("Updating support collections...");
		this.updateSupportCollections(ur);
		// record the group id
		this.addConsumerGroup(ur.getFullyQualifiedConsumerId());
	}


	private void updateSupportCollections(UsageRecord ur){
		DBCollection collResourceTypes = getResourceTypesCollection();
		BasicDBObject resourceType = new BasicDBObject(DBField.RESOURCE_TYPE.toString(),ur.getResourceType());
		DBCursor cursorResourceType = collResourceTypes.find(resourceType);
		if(!cursorResourceType.hasNext())
			collResourceTypes.insert(resourceType);

		DBCollection collResourceScopes = getResourceScopesCollection();
		BasicDBObject resourceScope = new BasicDBObject(DBField.RESOURCE_SCOPE.toString(),ur.getResourceScope());
		DBCursor cursorResourceScope = collResourceScopes.find(resourceScope);
		if(!cursorResourceScope.hasNext())
			collResourceScopes.insert(resourceScope);
	}

	private void updateStatistics(UsageRecord ur){
		DBCollection collResourceTypes = getResourceTypesCollection();
		BasicDBObject resourceType = new BasicDBObject(DBField.RESOURCE_TYPE.toString(),ur.getResourceType());
		DBCursor cursorResourceType = collResourceTypes.find(resourceType);
		if(!cursorResourceType.hasNext())
			collResourceTypes.insert(resourceType);

		DBCollection collResourceScopes = getResourceScopesCollection();
		BasicDBObject resourceScope = new BasicDBObject(DBField.RESOURCE_SCOPE.toString(),ur.getResourceScope());
		DBCursor cursorResourceScope = collResourceScopes.find(resourceScope);
		if(!cursorResourceScope.hasNext())
			collResourceScopes.insert(resourceScope);
	}

	private DBCollection ensureIndexesForUsageRecords(DBCollection usageRecordsCollection){
		//DBCollection coll = this.db.getCollection(DBCollections.USAGE_RECORDS.toString());
		usageRecordsCollection.ensureIndex(new BasicDBObject(DBField.UR_ID.toString(), 1).append("unique", true).append("dropDups", true));
		usageRecordsCollection.ensureIndex(new BasicDBObject(DBField.FULLY_QUALIFIED_CONSUMER_ID.toString(), 1));
		usageRecordsCollection.ensureIndex(new BasicDBObject(DBField.CONSUMER_ID.toString(), 1));
		usageRecordsCollection.ensureIndex(new BasicDBObject(DBField.END_TIME.toString(), 1));
		usageRecordsCollection.ensureIndex(new BasicDBObject(DBField.START_TIME.toString(), 1));
		usageRecordsCollection.ensureIndex(new BasicDBObject(DBField.CREATE_TIME.toString(), 1));
		usageRecordsCollection.ensureIndex(new BasicDBObject(DBField.RESOURCE_OWNER.toString(), 1));
		usageRecordsCollection.ensureIndex(new BasicDBObject(DBField.RESOURCE_SCOPE.toString(), 1));
		usageRecordsCollection.ensureIndex(new BasicDBObject(DBField.RESOURCE_TYPE.toString(), 1));
		return usageRecordsCollection;
	}

	private DBCollection getUsageRecordsCollection() {
		DBCollection coll = this.db.getCollection(DBCollections.USAGE_RECORDS.toString());
		logger.debug("enforcing indexes");
		return ensureIndexesForUsageRecords(coll);
	}

	private DBCollection getResourceTypesCollection() {
		DBCollection coll = this.db.getCollection(DBCollections.RESOURCE_TYPES.toString());
		logger.debug("enforcing indexes");
		//		coll.ensureIndex(new BasicDBObject(DBField.UR_ID.toString(), 1).append("unique", true).append("dropDups", true));
		//		coll.ensureIndex(new BasicDBObject(DBField.FULLY_QUALIFIED_CONSUMER_ID.toString(), 1));
		return coll;
	}

	private DBCollection getResourceScopesCollection() {
		DBCollection coll = this.db.getCollection(DBCollections.RESOURCE_SCOPES.toString());
		logger.debug("enforcing indexes");
		//		coll.ensureIndex(new BasicDBObject(DBField.UR_ID.toString(), 1).append("unique", true).append("dropDups", true));
		//		coll.ensureIndex(new BasicDBObject(DBField.FULLY_QUALIFIED_CONSUMER_ID.toString(), 1));
		return coll;
	}

	private DBCollection getStatisticsCollection() {
		DBCollection coll = this.db.getCollection(DBCollections.RESOURCE_TYPES.toString());
		logger.debug("enforcing indexes");
		//		coll.ensureIndex(new BasicDBObject(DBField.UR_ID.toString(), 1).append("unique", true).append("dropDups", true));
		//		coll.ensureIndex(new BasicDBObject(DBField.FULLY_QUALIFIED_CONSUMER_ID.toString(), 1));
		return coll;
	}

	/**
	 * Update the given UR (replacement based on id)
	 * @param ur
	 */
	public void updateUsageRecord(UsageRecord ur) {
		DBCollection coll = getUsageRecordsCollection();
		try {
			// retrieving existing record
			UsageRecord oldUR = this.getUsageRecord(ur.getId());
			// retrieving current consumer group (might not be in new record)
			// NO: updating group is allowed
			// ur.setFullyQualifiedConsumerId(oldUR.getFullyQualifiedConsumerId());
			// avoid clients to change the create time
			ur.setCreateTime(oldUR.getCreateTime());
			// update modify time (to 'now')
			//ur.setModifyTime(Calendar.getInstance());
			// update object
			DBObject newUR = this.createDBObject(ur);
			logger.info(String.format("updating record %s", ur.getId()));
			coll.update(new BasicDBObject().append("urId", ur.getId()), newUR);
		}
		catch(Exception e) {
			logger.error(e.getMessage(), e);
			e.printStackTrace();
		}
	}

	/**
	 * Retrieve URs with the given <key,value> pair
	 * @param key
	 * @param value
	 * @return
	 */
	/*
	public Collection<RawUsageRecord> getUsageRecordByProperty(String key, String value) {
		Collection<RawUsageRecord> out = new Vector<RawUsageRecord>();
		BasicDBObject query = new BasicDBObject();
		query.put(key, value);
		DBCollection coll = this.db.getCollection(USAGE_RECORDS);
		DBCursor cur = coll.find(query);
		while(cur.hasNext()) {
			DBObject dbo = cur.next();
			out.add(this.createUsageRecord(dbo));
		}
		return out;
	}
	 */

	/*
	public Collection<RawUsageRecord> getUsageRecordByProperties(Map<String, String> props) {
		Collection<RawUsageRecord> out = new Vector<RawUsageRecord>();
		BasicDBObject query = new BasicDBObject();
		for(String key:props.keySet()) {
			logger.debug("adding " + key + ":"+props.get(key));
			query.put(key, props.get(key));
		}
		DBCollection coll = this.db.getCollection(USAGE_RECORDS);
		DBCursor cur = coll.find(query);
		while(cur.hasNext()) {
			DBObject dbo = cur.next();
			out.add(this.createUsageRecord(dbo));
		}
		return out;
	}
	 */

	private BasicDBObject getDBObject(String key, String op, String value) {
		try {
			Long d = Long.parseLong(value);
			return new BasicDBObject(op, d);
		}
		catch(NumberFormatException e) {
			try {
				Double d = Double.parseDouble(value);
				return new BasicDBObject(op, d);
			}
			catch(NumberFormatException e2) {
				return new BasicDBObject(op, value);
			}
		}
	}


	private EnhancedQuery generateQuery (Map<String, QueryClause> props){
		BasicDBObject query = new BasicDBObject();
		ResultOptions options = new ResultOptions(DBField.END_TIME.toString());
		logger.debug("after enhanced query preliminary options setting");
		for(QueryClause clause:props.values()) {
			logger.debug("adding " + clause.getKey() + ":" + clause.getValue());
			if(QueryOperator.eq.equals(clause.getOperator())) {
				//special case for sorting and pagination
				if(clause.getKey().equals(AggregationField.SORT_FIELD.toString())){
					logger.debug("generating query with sortField = "+clause.getValue());
					options.setSortField(clause.getValue());
				}
				else
					if(clause.getKey().equals(AggregationField.SORT_ORDER.toString())){
						options.setSortOrder(clause.getValue());
					}
					else
						if(clause.getKey().equals(AggregationField.PAGINATION_START_RECORD.toString())){
							options.setStartRecord(Integer.valueOf(clause.getValue()));
						}
						else
							if(clause.getKey().equals(AggregationField.PAGINATION_RECORDS.toString())){
								options.setRecordsPerPage(Integer.valueOf(clause.getValue()));
							}
							else
								if(clause.getKey().equals("startTime")){
									StringTokenizer st = new StringTokenizer(clause.getValue(), ",");
									String from = st.nextToken();
									String to = st.nextToken();
									logger.debug("from = "+from);
									logger.debug("to = "+to);

									BasicDBObject date = new BasicDBObject();
									DateFormat df = new SimpleDateFormat("yyyy-MM-dd");
									Date fromDate = null;
									Date toDate = null;
									try {
										fromDate = df.parse(from);
										toDate = df.parse(to);
									} catch (ParseException e) {
										// TODO Auto-generated catch block
										e.printStackTrace();
									}

									date.append("$gte", fromDate);
									date.append("$lte", toDate);

									query.put(clause.getKey(),date);
									logger.debug("query = "+query.toMap());
								}
								else

									//special case for consumer group
									if(clause.getKey().equals(DBField.FULLY_QUALIFIED_CONSUMER_ID.toString())){
										// ensure index on consumerGroup
										//DBCollection coll = this.getRecordsCollection();

										// prepare clause

										//String pattern = "^"+clause.getValue().replace(".", ",");
										String pattern = "^"+clause.getValue().replace(".", "\\.");
										Pattern match = Pattern.compile(pattern);
										query.put(clause.getKey(), match);

									}
									else
										if(clause.getValue().contains(",")){
											List<String> listValues = new ArrayList<String>();
											StringTokenizer stMultipleValues = new StringTokenizer(clause.getValue(), ",");
											while(stMultipleValues.hasMoreTokens()) {
												String kvMultipleValues = stMultipleValues.nextToken();
												listValues.add(kvMultipleValues);
											}
											query.put(clause.getKey(), new BasicDBObject("$in",listValues));
										}
										else
											{
											try {
												Long d = Long.parseLong(clause.getValue());
												logger.debug("Long ------" +clause.getKey()+", "+clause.getValue());
												query.put(clause.getKey(), d);
											}
											catch(NumberFormatException e) {
												try {
													Double d = Double.parseDouble(clause.getValue());
													logger.debug("Double ------" +clause.getKey()+", "+clause.getValue());
													query.put(clause.getKey(), d);
												}
												catch(NumberFormatException e2) {
													query.put(clause.getKey(), clause.getValue());
												}
											}
										}
			}
			else if(QueryOperator.lt.equals(clause.getOperator())) {
				query.put(clause.getKey(), getDBObject(clause.getKey(), "$lt", clause.getValue()));
			}
			else if(QueryOperator.lte.equals(clause.getOperator())) {
				query.put(clause.getKey(), getDBObject(clause.getKey(), "$lte", clause.getValue()));
			}
			else if(QueryOperator.gt.equals(clause.getOperator())) {
				query.put(clause.getKey(), getDBObject(clause.getKey(), "$gt", clause.getValue()));
			}
			else if(QueryOperator.gte.equals(clause.getOperator())) {
				query.put(clause.getKey(), getDBObject(clause.getKey(), "$gte", clause.getValue()));
			}
			else {
				logger.warn("operator '" + clause.getOperator() + "' is not yet supported");
			}
		}
		return new EnhancedQuery(query,options);
	}

	/**
	 * Computes the nr of URs in the store
	 * @return
	 */
	public long size() {
		return this.getUsageRecordsCollection().count();
		//		return db.getCollection(USAGE_RECORDS).count();
	}

	/**
	 * Computes the nr of URs in the store against a specific query
	 * @return
	 */
	public long size(Map<String, QueryClause> props) {
		EnhancedQuery query = generateQuery(props);
		//		logger.debug("Query inside size method = "+query.getQuery().toString());
		return this.getUsageRecordsCollection().find(query.getQuery()).count();
		//		return db.getCollection(USAGE_RECORDS).count();
	}

	public Collection<RawUsageRecord> getUsageRecordByQuery(Map<String, QueryClause> props) {
		Collection<RawUsageRecord> out = new Vector<RawUsageRecord>();

		DBCollection coll = null;
		DBCursor cursor = null;
		coll = this.db.getCollection(DBCollections.USAGE_RECORDS.toString());
		logger.debug("Befor enhanced query");
		EnhancedQuery enhancedQuery = generateQuery(props);


		BasicDBObject sortCriteria;
		sortCriteria = new BasicDBObject(enhancedQuery.getOptions().getSortField(), enhancedQuery.getOptions().getSortOrder()); 

		logger.debug("sortCriteria = "+enhancedQuery.getOptions().getSortField()+" , "+enhancedQuery.getOptions().getSortOrder());

		logger.debug("** enhancedQuery = "+enhancedQuery.getQuery());
		cursor = coll.find(enhancedQuery.getQuery()).sort(sortCriteria)
				.skip(enhancedQuery.getOptions().getStartRecord()).
				limit(enhancedQuery.getOptions().getRecordsPerPage());

		logger.debug("found " + cursor.size() + " records");
		while(cursor.hasNext()) {
			DBObject dbo = cursor.next();
			out.add(this.createUsageRecord(dbo));
		}
		return out;
	}


	/**
	 * Return all records _entirely_ falling in the given time interval
	 * @param user
	 * @param fromDate
	 * @param toDate
	 * @return
	 */
	/*
	public Collection<RawUsageRecord> getUsageRecordsByUserByTime(String user, Calendar fromDate, Calendar toDate) {
		Collection<RawUsageRecord> out = new Vector<RawUsageRecord>();

		logger.info("looking for URs between " + fromDate.getTimeInMillis() + " and " + toDate.getTimeInMillis());

		// 1) completely inside the window
		BasicDBObject query = new BasicDBObject();
		query.put(CONSUMER_ID, user);
		query.put(START_TIME, new BasicDBObject("$gt", fromDate.getTimeInMillis()));
		query.put(END_TIME, new BasicDBObject("$lt", toDate.getTimeInMillis()));
		DBCollection coll = this.db.getCollection(USAGE_RECORDS);
		DBCursor cur = coll.find(query);
		while(cur.hasNext()) {
			DBObject dbo = cur.next();
			out.add(this.createUsageRecord(dbo));
		}

		// 2) completely covering the window
		query = new BasicDBObject();
		query.put(CONSUMER_ID, user);
		query.put(START_TIME, new BasicDBObject("$lt", fromDate.getTimeInMillis()));
		query.put(END_TIME, new BasicDBObject("$gt", toDate.getTimeInMillis()));
		coll = this.db.getCollection(USAGE_RECORDS);
		cur = coll.find(query);
		while(cur.hasNext()) {
			DBObject dbo = cur.next();
			out.add(this.createUsageRecord(dbo));
		}

		// 3) ending in the period
		query = new BasicDBObject();
		query.put(CONSUMER_ID, user);
		query.put(START_TIME, new BasicDBObject("$lt", fromDate.getTimeInMillis()));
		query.put(END_TIME, new BasicDBObject("$gt", fromDate.getTimeInMillis()).append("$lt", toDate.getTimeInMillis()));
		coll = this.db.getCollection(USAGE_RECORDS);
		cur = coll.find(query);
		while(cur.hasNext()) {
			DBObject dbo = cur.next();
			out.add(this.createUsageRecord(dbo));
		}

		// 4) started in the period
		query = new BasicDBObject();
		query.put(CONSUMER_ID, user);
		query.put(START_TIME, new BasicDBObject("$gt", fromDate.getTimeInMillis()).append("$lt", toDate.getTimeInMillis()));
		query.put(END_TIME, new BasicDBObject("$gt", toDate.getTimeInMillis()));
		coll = this.db.getCollection(USAGE_RECORDS);
		cur = coll.find(query);
		while(cur.hasNext()) {
			DBObject dbo = cur.next();
			out.add(this.createUsageRecord(dbo));
		}

		return out;
	}
	 */

	/**
	 * Retrieve the UR with the given ID
	 */
	public UsageRecord getUsageRecord(String id) throws NotFoundException {
		BasicDBObject query = new BasicDBObject();
		query.put(DBField.UR_ID.toString(), id);
		DBCollection coll = this.db.getCollection(DBCollections.USAGE_RECORDS.toString());
		DBCursor cur = coll.find(query);
		if (cur.hasNext()) {
			DBObject out = cur.next();

			return this.createUsageRecord(out);
		}
		throw new NotFoundException("id");
	}

	/**
	 * Delete the UR with the given ID
	 * @param id
	 */
	public void deleteUsageRecord(String id) {
		logger.info(String.format("deleting usage record '%s' ...", id));
		BasicDBObject query = new BasicDBObject();
		query.put(DBField.UR_ID.toString(), id);
		DBCollection coll = this.db.getCollection(DBCollections.USAGE_RECORDS.toString());
		coll.remove(query, WriteConcern.SAFE);
	}

	/**
	 * Remove all Usage Records
	 */
	public void clear() {
		this.getTrackedUsers();
		// remove all records
		DBCollection coll = this.db.getCollection(DBCollections.USAGE_RECORDS.toString());
		BasicDBObject query = new BasicDBObject();
		coll.remove(query, WriteConcern.SAFE);
		// remove all groups
		coll = this.db.getCollection(DBCollections.CONSUMERS_RECORDS.toString());
		query = new BasicDBObject();
		coll.remove(query, WriteConcern.SAFE);
	}

	/**
	 * Return all usage records in the store
	 * @return
	 */

	public Collection<RawUsageRecord> getAll() {
		Collection<RawUsageRecord> out = new Vector<RawUsageRecord>();
		DBCollection coll = this.db.getCollection(DBCollections.USAGE_RECORDS.toString());
		BasicDBObject query = new BasicDBObject();
		DBCursor cur = coll.find(query);
		while(cur.hasNext()) {
			DBObject dbo = cur.next();
			out.add(this.createUsageRecord(dbo));
		}
		return out;
	}


	public Collection<String> getTrackedUsers() {
		return this.getDistinct(DBField.CONSUMER_ID.toString());
	}

	public Collection<String> getTrackedResourceTypes() {
		List<String> out = new Vector<String>();
		DBCollection coll = getResourceTypesCollection();
		DBCursor items = coll.find();
		while(items.hasNext()) {
			DBObject dbo = items.next();
			out.add((String) dbo.get(DBField.RESOURCE_TYPE.toString()));
		}
		return out;
	}

	public Collection<String> getTrackedResourceScopes() {
		List<String> out = new Vector<String>();
		DBCollection coll = getResourceScopesCollection();
		DBCursor items = coll.find();
		while(items.hasNext()) {
			DBObject dbo = items.next();
			out.add((String) dbo.get(DBField.RESOURCE_SCOPE.toString()));
		}
		return out;
	}

	public Collection<String> getTrackedResourceOwners() {
		return this.getDistinct(DBField.RESOURCE_OWNER.toString());
	}

	public Collection<String> getTrackedGroups() {
		return this.getDistinct(DBField.FULLY_QUALIFIED_CONSUMER_ID.toString());
	}

	@SuppressWarnings("rawtypes")
	private Collection<String> getDistinct(String key) {
		List<String> out = new Vector<String>();
		DBCollection coll = this.db.getCollection(DBCollections.USAGE_RECORDS.toString());
		List items = coll.distinct(key);
		for(Object o:items) {
			if(o!=null)
				out.add(o.toString());
		}
		return out;
	}

	/**
	 * Return the last update time for the given DCI.
	 * FIXME: for the moment the 'end_time' is used since we're using the simulator. Later on, we need to use the 'create_time'
	 * @param dci
	 * @return
	 */
	public Long getLastUpdate(String dci) {
		Long max = 0L;
		BasicDBObject query = new BasicDBObject();
		query.put(DBField.RESOURCE_OWNER.toString(), dci);
		DBCollection coll = this.db.getCollection(DBCollections.USAGE_RECORDS.toString());
		DBCursor cur = coll.find(query);
		while(cur.hasNext()) {
			DBObject dbo = cur.next();
			Long l = (Long)dbo.get(DBField.END_TIME.toString());
			if(l!=null && l>max)
				max = l;
		}
		return max;
	}

	public void addConsumerGroup(String consumerGroup) {
		List<String> l = new Vector<String>();
		StringTokenizer st = new StringTokenizer(consumerGroup, ".");
		while(st.hasMoreTokens()) {
			l.add(st.nextToken());
		}
		this.addConsumerGroup(l);
	}

	public void addConsumerGroup(List<String> consumerGroup) {
		logger.info("adding group: " + consumerGroup.toString());
		DBCollection coll = this.db.getCollection(DBCollections.CONSUMERS_RECORDS.toString());
		BasicDBObject query = new BasicDBObject();
		query.put(DBField.SUBSET_GROUP.toString(), consumerGroup);
		DBCursor cur = coll.find(query);
		if(!cur.hasNext()){
			BasicDBObject doc = new BasicDBObject();
			doc.put(DBField.SUBSET_GROUP.toString(), consumerGroup);
			coll.insert(doc);
		}	
	}

	public Collection<ArrayList<String>> getConsumersGroup(){
		Collection<ArrayList<String>> consumersGroup = new ArrayList<ArrayList<String>>();
		DBCollection collCG = this.db.getCollection(DBCollections.CONSUMERS_RECORDS.toString());
		collCG.ensureIndex(new BasicDBObject(DBField.SUBSET_GROUP.toString(), 1).append("unique", true));
		BasicDBObject query = new BasicDBObject();
		DBCursor cursor = collCG.find(query);
		while(cursor.hasNext()){
			DBObject dbo = cursor.next();
			BasicDBList dblist = (BasicDBList) dbo.get(DBField.SUBSET_GROUP.toString());
			ArrayList<String> current = new ArrayList<String>();
			for (Object object : dblist) {
				current.add((String) object);
			}
			consumersGroup.add(current);

		}
		return consumersGroup;
	}

	private String encodeKey(String key){
		key=key.replaceAll("\\\\", BACKSLASH);
		key=key.replaceAll("\\$", DOLLAR);
		key=key.replaceAll("\\.", DOT);
		return key;
	}

	private String decodeKey(String key){
		key=key.replaceAll(DOT, ".");
		key=key.replaceAll(DOLLAR,"\\$");
		key=key.replaceAll(BACKSLASH,"\\\\");
		return key;
	}


	//filters

	//task
	public FilterRating getTaskWorkerTimeFilter(Report report) {
		FilterRating out = new FilterRating();

		TreeSet<EntryFilterRating> rating = new TreeSet<EntryFilterRating>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,false);
		List<String> workingTime = new ArrayList<String>();
		workingTime.add("$rsp:usageEnd");
		workingTime.add("$rsp:usageStart");
		group.put("taskWorkerTime",  new BasicDBObject("$sum",new BasicDBObject("$subtract", workingTime)));
		DBObject groupClause = new BasicDBObject("$group", group);	

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause);

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 

			double volume = Double.parseDouble(obj.get("taskWorkerTime").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("taskWorkerTime = "+volume);
			rating.add(new EntryFilterRating(filter, volume));
		}
		out.setRating(rating);

		return out;
	}

	public FilterRating getTaskNetworkTrafficInputFilter(Report report) {
		FilterRating out = new FilterRating();

		TreeSet<EntryFilterRating> rating = new TreeSet<EntryFilterRating>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,false);
		group.put("taskNetworkTrafficInput",  new BasicDBObject("$sum","$"+report.getReportProperty()));
		DBObject groupClause = new BasicDBObject("$group", group);	


		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause);

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 

			double networkTrafficIn = Double.parseDouble(obj.get("taskNetworkTrafficInput").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("networkTrafficIn = "+networkTrafficIn);
			rating.add(new EntryFilterRating(filter, networkTrafficIn));
		}
		out.setRating(rating);

		return out;
	}

	public FilterRating getTaskNetworkTrafficOutputFilter(Report report) {
		FilterRating out = new FilterRating();

		TreeSet<EntryFilterRating> rating = new TreeSet<EntryFilterRating>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,false);
		group.put("taskNetworkTrafficOutput",  new BasicDBObject("$sum","$"+report.getReportProperty()));
		DBObject groupClause = new BasicDBObject("$group", group);	


		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause);

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 

			double volume = Double.parseDouble(obj.get("taskNetworkTrafficOutput").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("count = "+volume);
			rating.add(new EntryFilterRating(filter, volume));
		}
		out.setRating(rating);

		return out;
	}

	public FilterRating getTaskCountProcessorsFilter(Report report) {
		FilterRating out = new FilterRating();

		TreeSet<EntryFilterRating> rating = new TreeSet<EntryFilterRating>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,false);
		group.put("taskCountProcessors",  new BasicDBObject("$sum","$rsp:processors"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause);

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 

			double volume = Double.parseDouble(obj.get("taskCountProcessors").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("count = "+volume);
			rating.add(new EntryFilterRating(filter, volume));
		}
		out.setRating(rating);

		return out;
	}

	//job
	public FilterRating getJobNetElapsedTimeFilter(Report report) {
		FilterRating out = new FilterRating();

		TreeSet<EntryFilterRating> rating = new TreeSet<EntryFilterRating>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,false);

		group.put("jobNetElapsedTime",  new BasicDBObject("$sum","$rsp:wallDuration"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause);

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 

			double volume = Double.parseDouble(obj.get("jobNetElapsedTime").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("jobNetElapsedTime = "+volume);
			rating.add(new EntryFilterRating(filter, volume));
		}
		out.setRating(rating);

		return out;
	}

	public FilterRating getJobElapsedTimeFilter(Report report) {
		FilterRating out = new FilterRating();

		TreeSet<EntryFilterRating> rating = new TreeSet<EntryFilterRating>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,false);
		List<String> workingTime = new ArrayList<String>();
		workingTime.add("$rsp:jobEnd");
		workingTime.add("$rsp:jobStart");
		group.put("jobElapsedTime",  new BasicDBObject("$sum",new BasicDBObject("$subtract", workingTime)));
		DBObject groupClause = new BasicDBObject("$group", group);	

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause);

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 

			double volume = Double.parseDouble(obj.get("jobElapsedTime").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("jobElapsedTime = "+volume);
			rating.add(new EntryFilterRating(filter, volume));
		}
		out.setRating(rating);

		return out;
	}

	public FilterRating getJobCountJobsFilter(Report report) {
		FilterRating out = new FilterRating();

		TreeSet<EntryFilterRating> rating = new TreeSet<EntryFilterRating>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,false);
		group.put("jobCountJobs", new BasicDBObject("$sum",1));
		DBObject groupClause = new BasicDBObject("$group", group);	

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause);

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 

			double volume = Double.parseDouble(obj.get("jobCountJobs").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("count = "+volume);
			rating.add(new EntryFilterRating(filter, volume));
		}
		out.setRating(rating);

		return out;
	}

	//service
	public FilterRating getServiceCountOperationsFilter(Report report) {
		FilterRating out = new FilterRating();

		TreeSet<EntryFilterRating> rating = new TreeSet<EntryFilterRating>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,false);
		group.put("serviceCountOperations", new BasicDBObject("$sum","$rsp:invocationCount"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause);

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 

			double volume = Double.parseDouble(obj.get("serviceCountOperations").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("volume = "+volume);
			rating.add(new EntryFilterRating(filter, volume));
		}
		out.setRating(rating);

		return out;
	}

	public FilterRating getServiceAvgInvocationTimeFilter(Report report) {
		FilterRating out = new FilterRating();

		TreeSet<EntryFilterRating> rating = new TreeSet<EntryFilterRating>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,false);
		group.put("serviceAvgInvocationTime", new BasicDBObject("$avg","$rsp:averageInvocationTime"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause);

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 

			double volume = Double.parseDouble(obj.get("serviceAvgInvocationTime").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("volume = "+volume);
			rating.add(new EntryFilterRating(filter, volume));
		}
		out.setRating(rating);

		return out;
	}

	//storage-status
	public FilterRating getStorageStatusAvgEntitiesFilter(Report report) {
		FilterRating out = new FilterRating();

		TreeSet<EntryFilterRating> rating = new TreeSet<EntryFilterRating>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,false);
		group.put("storageStatusAvgEntities", new BasicDBObject("$avg","$rsp:dataCount"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause);

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 

			double volume = Double.parseDouble(obj.get("storageStatusAvgEntities").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("storageStatusAvgEntities = "+volume);
			rating.add(new EntryFilterRating(filter, volume));
		}
		out.setRating(rating);

		return out;
	}

	public FilterRating getStorageStatusAvgVolumeFilter(Report report) {
		FilterRating out = new FilterRating();

		TreeSet<EntryFilterRating> rating = new TreeSet<EntryFilterRating>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,false);
		group.put("storageStatusAvgVolume", new BasicDBObject("$avg","$rsp:dataVolume"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause);

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 

			double volume = Double.parseDouble(obj.get("storageStatusAvgVolume").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("storageStatusAvgVolume = "+volume);
			rating.add(new EntryFilterRating(filter, volume));
		}
		out.setRating(rating);

		return out;
	}

	public FilterRating getStorageStatusVolumeFilter(Report report) {
		FilterRating out = new FilterRating();

		TreeSet<EntryFilterRating> rating = new TreeSet<EntryFilterRating>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,false);
		group.put("storageStatusVolume", new BasicDBObject("$last","$rsp:dataVolume"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause);

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 

			double volume = Double.parseDouble(obj.get("storageStatusVolume").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("storageStatusVolume = "+volume);
			rating.add(new EntryFilterRating(filter, volume));
		}
		out.setRating(rating);

		return out;
	}

	public FilterRating getStorageStatusEntitiesFilter(Report report) {
		FilterRating out = new FilterRating();
		TreeSet<EntryFilterRating> rating = new TreeSet<EntryFilterRating>();
		DBObject group = MongoAggregationHelper.generateGroupClause(report,false);
		group.put("storageStatusEntities", new BasicDBObject("$avg","$rsp:dataCount"));
		DBObject groupClause = new BasicDBObject("$group", group);	
		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause);
		for (DBObject obj : output.results()) {
			DBObject idField = (DBObject) obj.get("_id"); 
			double volume = Double.parseDouble(obj.get("storageStatusEntities").toString());
			String filter = idField.get(report.getDimension()).toString();
			logger.debug("storageStatusEntities = "+volume);
			rating.add(new EntryFilterRating(filter, volume));
		}
		out.setRating(rating);
		return out;
	}

	//storage-usage
	public FilterRating getStorageUsageVolumeFilter(Report report) {
		FilterRating out = new FilterRating();
		TreeSet<EntryFilterRating> rating = new TreeSet<EntryFilterRating>();
		DBObject group = MongoAggregationHelper.generateGroupClause(report,false);
		group.put("storageUsageVolume", new BasicDBObject("$sum","$rsp:dataVolume"));
		DBObject groupClause = new BasicDBObject("$group", group);	
		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause);
		for (DBObject obj : output.results()) {
			DBObject idField = (DBObject) obj.get("_id"); 
			double volume = Double.parseDouble(obj.get("storageUsageVolume").toString());
			String filter = idField.get(report.getDimension()).toString();
			logger.debug("count = "+volume);
			rating.add(new EntryFilterRating(filter, volume));
		}
		out.setRating(rating);
		return out;
	}

	public FilterRating getStorageUsageCountEntitiesFilter(Report report) {
		FilterRating out = new FilterRating();

		TreeSet<EntryFilterRating> rating = new TreeSet<EntryFilterRating>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,false);
		group.put("storageUsageCountEntities", new BasicDBObject("$sum","$rsp:dataCount"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause);

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 

			double volume = Double.parseDouble(obj.get("storageUsageCountEntities").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("count = "+volume);
			rating.add(new EntryFilterRating(filter, volume));
		}
		out.setRating(rating);

		return out;
	}

	public FilterRating getStorageUsageCountOperationsFilter(Report report) {
		FilterRating out = new FilterRating();

		TreeSet<EntryFilterRating> rating = new TreeSet<EntryFilterRating>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,false);
		group.put("storageUsageCountOperations", new BasicDBObject("$avg","$rsp:dataCount"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause);

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 

			double volume = Double.parseDouble(obj.get("storageUsageCountOperations").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("count = "+volume);
			rating.add(new EntryFilterRating(filter, volume));
		}
		out.setRating(rating);

		return out;
	}






	//reports

	//task
	public Collection<AggregatedResult> getTaskNetworkTrafficOutputReport(Report report) {
		Collection<AggregatedResult> out = new ArrayList<AggregatedResult>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,true);
		group.put("taskNetworkTrafficOutput",  new BasicDBObject("$sum","$"+report.getReportProperty()));
		DBObject groupClause = new BasicDBObject("$group", group);	

		logger.debug("groupClause = "+groupClause);

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause,MongoAggregationHelper.generateSortClause(report));

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 
			DateTime time = MongoAggregationHelper.generateAggregationTime(idField, report.getAggregationGranule());

			double volume = Double.parseDouble(obj.get("taskNetworkTrafficOutput").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("dimension = "+filter+ " - "+ "taskNetworkTrafficOutput = "+volume);

			AggregatedResult current = new AggregatedResult(time, Metric.SUM, volume, filter);
			out.add(current);
		}

		Collections.sort((List<AggregatedResult>) out);
		return out;
	}

	public Collection<AggregatedResult> getTaskNetworkTrafficInputReport(Report report) {
		Collection<AggregatedResult> out = new ArrayList<AggregatedResult>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,true);
		group.put("taskNetworkTrafficInput",  new BasicDBObject("$sum","$"+report.getReportProperty()));
		DBObject groupClause = new BasicDBObject("$group", group);	

		logger.debug("groupClause = "+groupClause);

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause,MongoAggregationHelper.generateSortClause(report));

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 
			DateTime time = MongoAggregationHelper.generateAggregationTime(idField, report.getAggregationGranule());

			double volume = Double.parseDouble(obj.get("taskNetworkTrafficInput").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("dimension = "+filter+ " - "+ "taskNetworkTrafficInput = "+volume);

			AggregatedResult current = new AggregatedResult(time, Metric.SUM, volume, filter);
			out.add(current);
		}

		Collections.sort((List<AggregatedResult>) out);
		return out;
	}

	public Collection<AggregatedResult> getTaskWorkerTimeReport(Report report) {
		Collection<AggregatedResult> out = new ArrayList<AggregatedResult>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,true);
		List<String> workingTime = new ArrayList<String>();
		workingTime.add("$rsp:usageEnd");
		workingTime.add("$rsp:usageStart");
		group.put("taskWorkerTime",  new BasicDBObject("$sum",new BasicDBObject("$subtract", workingTime)));
		DBObject groupClause = new BasicDBObject("$group", group);	

		logger.debug("groupClause = "+groupClause);

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause,MongoAggregationHelper.generateSortClause(report));

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 
			DateTime time = MongoAggregationHelper.generateAggregationTime(idField, report.getAggregationGranule());

			double volume = Double.parseDouble(obj.get("taskWorkerTime").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("dimension = "+filter+ " - "+ "taskWorkerTime = "+volume);

			AggregatedResult current = new AggregatedResult(time, Metric.SUM, volume, filter);
			out.add(current);
		}

		Collections.sort((List<AggregatedResult>) out);
		return out;
	}

	public Collection<AggregatedResult> getTaskCountProcessorsReport(Report report) {
		Collection<AggregatedResult> out = new ArrayList<AggregatedResult>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,true);
		group.put("taskCountProcessors",  new BasicDBObject("$sum","$rsp:processors"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		logger.debug("groupClause = "+groupClause);

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause,MongoAggregationHelper.generateSortClause(report));

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 
			DateTime time = MongoAggregationHelper.generateAggregationTime(idField, report.getAggregationGranule());

			double volume = Double.parseDouble(obj.get("taskCountProcessors").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("dimension = "+filter+ " - "+ "taskCountProcessors = "+volume);

			AggregatedResult current = new AggregatedResult(time, Metric.SUM, volume, filter);
			out.add(current);
		}

		Collections.sort((List<AggregatedResult>) out);
		return out;
	}

	//job
	public Collection<AggregatedResult> getJobNetElapsedTimeReport(Report report) {
		Collection<AggregatedResult> out = new ArrayList<AggregatedResult>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,true);
		group.put("jobNetElapsedTime",  new BasicDBObject("$sum","$rsp:wallDuration"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		logger.debug("groupClause = "+groupClause);

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause,MongoAggregationHelper.generateSortClause(report));

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 
			DateTime time = MongoAggregationHelper.generateAggregationTime(idField, report.getAggregationGranule());

			double volume = Double.parseDouble(obj.get("jobNetElapsedTime").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("dimension = "+filter+ " - "+ "jobNetElapsedTime = "+volume);

			AggregatedResult current = new AggregatedResult(time, Metric.SUM, volume, filter);
			out.add(current);
		}

		Collections.sort((List<AggregatedResult>) out);
		return out;
	}

	public Collection<AggregatedResult> getJobElapsedTimeReport(Report report) {
		Collection<AggregatedResult> out = new ArrayList<AggregatedResult>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,true);
		List<String> workingTime = new ArrayList<String>();
		workingTime.add("$rsp:jobEnd");
		workingTime.add("$rsp:jobStart");
		group.put("jobElapsedTime",  new BasicDBObject("$sum",new BasicDBObject("$subtract", workingTime)));
		DBObject groupClause = new BasicDBObject("$group", group);	

		logger.debug("groupClause = "+groupClause);

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause,MongoAggregationHelper.generateSortClause(report));

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 
			DateTime time = MongoAggregationHelper.generateAggregationTime(idField, report.getAggregationGranule());

			double volume = Double.parseDouble(obj.get("jobElapsedTime").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("dimension = "+filter+ " - "+ "jobElapsedTime = "+volume);

			AggregatedResult current = new AggregatedResult(time, Metric.SUM, volume, filter);
			out.add(current);
		}

		Collections.sort((List<AggregatedResult>) out);
		return out;
	}

	public Collection<AggregatedResult> getJobCountJobsReport(Report report) {
		Collection<AggregatedResult> out = new ArrayList<AggregatedResult>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,true);
		group.put("jobCountJobs", new BasicDBObject("$sum",1));
		DBObject groupClause = new BasicDBObject("$group", group);	

		logger.debug("groupClause = "+groupClause);

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause,MongoAggregationHelper.generateSortClause(report));

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 
			DateTime time = MongoAggregationHelper.generateAggregationTime(idField, report.getAggregationGranule());

			double volume = Double.parseDouble(obj.get("jobCountJobs").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("dimension = "+filter+ " - "+ "jobCountJobs = "+volume);

			AggregatedResult current = new AggregatedResult(time, Metric.SUM, volume, filter);
			out.add(current);
		}

		Collections.sort((List<AggregatedResult>) out);
		return out;
	}


	//service
	public Collection<AggregatedResult> getServiceCountOperationsReport(Report report) {
		Collection<AggregatedResult> out = new ArrayList<AggregatedResult>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,true);
		group.put("serviceCountOperations", new BasicDBObject("$sum","$rsp:invocationCount"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		logger.debug("groupClause = "+groupClause);

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause,MongoAggregationHelper.generateSortClause(report));

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 
			DateTime time = MongoAggregationHelper.generateAggregationTime(idField, report.getAggregationGranule());

			double volume = Double.parseDouble(obj.get("serviceCountOperations").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("dimension = "+filter+ " - "+ "serviceCountOperations = "+volume);

			AggregatedResult current = new AggregatedResult(time, Metric.SUM, volume, filter);
			out.add(current);
		}

		Collections.sort((List<AggregatedResult>) out);
		return out;
	}

	public Collection<AggregatedResult> getServiceAvgInvocationTimeReport(Report report) {
		Collection<AggregatedResult> out = new ArrayList<AggregatedResult>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,true);
		group.put("serviceAvgInvocationTime", new BasicDBObject("$avg","$rsp:averageInvocationTime"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		logger.debug("groupClause = "+groupClause);

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause,MongoAggregationHelper.generateSortClause(report));

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 
			DateTime time = MongoAggregationHelper.generateAggregationTime(idField, report.getAggregationGranule());

			double volume = Double.parseDouble(obj.get("serviceAvgInvocationTime").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("dimension = "+filter+ " - "+ "serviceAvgInvocationTime = "+volume);

			AggregatedResult current = new AggregatedResult(time, Metric.AVG, volume, filter);
			out.add(current);
		}

		Collections.sort((List<AggregatedResult>) out);
		return out;
	}


	//storage-usage
	public Collection<AggregatedResult> getStorageUsageVolumeReport(Report report) {
		Collection<AggregatedResult> out = new ArrayList<AggregatedResult>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,true);
		group.put("storageUsageVolume", new BasicDBObject("$sum","$rsp:dataVolume"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause,MongoAggregationHelper.generateSortClause(report));

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 
			DateTime time = MongoAggregationHelper.generateAggregationTime(idField, report.getAggregationGranule());

			double dataCount = Double.parseDouble(obj.get("storageUsageVolume").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("dimension = "+filter+ " - "+ "storageUsageVolume = "+dataCount);

			AggregatedResult current = new AggregatedResult(time, Metric.SUM, dataCount, filter);
			out.add(current);
		}

		Collections.sort((List<AggregatedResult>) out);
		return out;
	}

	public Collection<AggregatedResult> getStorageUsageCountEntitiesReport(Report report) {
		Collection<AggregatedResult> out = new ArrayList<AggregatedResult>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,true);
		group.put("storageUsageCountEntities", new BasicDBObject("$sum","$rsp:dataCount"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause,MongoAggregationHelper.generateSortClause(report));

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 
			DateTime time = MongoAggregationHelper.generateAggregationTime(idField, report.getAggregationGranule());

			double dataCount = Double.parseDouble(obj.get("storageUsageCountEntities").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("dimension = "+filter+ " - "+ "storageUsageCountEntities = "+dataCount);

			AggregatedResult current = new AggregatedResult(time, Metric.SUM, dataCount, filter);
			out.add(current);
		}

		Collections.sort((List<AggregatedResult>) out);
		return out;
	}

	public Collection<AggregatedResult> getStorageUsageCountOperationsReport(Report report) {
		Collection<AggregatedResult> out = new ArrayList<AggregatedResult>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,true);
		group.put("storageUsageCountOperations", new BasicDBObject("$sum","$rsp:dataCount"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause,MongoAggregationHelper.generateSortClause(report));

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 
			DateTime time = MongoAggregationHelper.generateAggregationTime(idField, report.getAggregationGranule());

			double dataCount = Double.parseDouble(obj.get("storageUsageCountOperations").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("dimension = "+filter+ " - "+ "storageUsageCountOperations = "+dataCount);

			AggregatedResult current = new AggregatedResult(time, Metric.COUNT, dataCount, filter);
			out.add(current);
		}

		Collections.sort((List<AggregatedResult>) out);
		return out;
	}

	//storage-status
	public Collection<AggregatedResult> getStorageStatusEntitiesReport(Report report) {
		Collection<AggregatedResult> out = new ArrayList<AggregatedResult>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,true);
		group.put("storageStatusEntities", new BasicDBObject("$last","$rsp:dataCount"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause,MongoAggregationHelper.generateSortClause(report));

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 
			DateTime time = MongoAggregationHelper.generateAggregationTime(idField, report.getAggregationGranule());

			double dataVolume = Double.parseDouble(obj.get("storageStatusEntities").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("dimension = "+filter+ " - "+ "storageStatusEntities = "+dataVolume);

			AggregatedResult current = new AggregatedResult(time, Metric.COUNT, dataVolume, filter);
			out.add(current);
		}

		Collections.sort((List<AggregatedResult>) out);
		return out;
	}

	public Collection<AggregatedResult> getStorageStatusVolumeReport(Report report) {
		Collection<AggregatedResult> out = new ArrayList<AggregatedResult>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,true);
		group.put("storageStatusVolume", new BasicDBObject("$last","$rsp:dataVolume"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause,MongoAggregationHelper.generateSortClause(report));

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 
			DateTime time = MongoAggregationHelper.generateAggregationTime(idField, report.getAggregationGranule());

			double dataVolume = Double.parseDouble(obj.get("storageStatusVolume").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("dimension = "+filter+ " - "+ "storageStatusVolume = "+dataVolume);

			AggregatedResult current = new AggregatedResult(time, Metric.COUNT, dataVolume, filter);
			out.add(current);
		}

		Collections.sort((List<AggregatedResult>) out);
		return out;
	}


	public Collection<AggregatedResult> getStorageStatusAvgEntitiesReport(Report report) {
		Collection<AggregatedResult> out = new ArrayList<AggregatedResult>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,true);
		group.put("storageStatusAvgEntities", new BasicDBObject("$avg","$rsp:dataCount"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause,MongoAggregationHelper.generateSortClause(report));

		for (DBObject obj : output.results()) {


			DBObject idField = (DBObject) obj.get("_id"); 
			DateTime time = MongoAggregationHelper.generateAggregationTime(idField, report.getAggregationGranule());

			double volume = Double.parseDouble(obj.get("storageStatusAvgEntities").toString());

			String filter = idField.get(report.getDimension()).toString();
			logger.debug("dimension = "+filter+ " - "+ "avgEntities = "+volume);

			AggregatedResult current = new AggregatedResult(time, Metric.AVG, volume, filter);
			out.add(current);
		}

		Collections.sort((List<AggregatedResult>) out);
		return out;


	}


	public Collection<AggregatedResult> getStorageStatusAvgVolumeReport(Report report) {
		Collection<AggregatedResult> out = new ArrayList<AggregatedResult>();

		DBObject group = MongoAggregationHelper.generateGroupClause(report,true);
		group.put("storageStatusAvgVolume", new BasicDBObject("$avg","$rsp:dataVolume"));
		DBObject groupClause = new BasicDBObject("$group", group);	

		AggregationOutput output = this.getUsageRecordsCollection().aggregate(MongoAggregationHelper.generateMatchClause(report), groupClause,MongoAggregationHelper.generateSortClause(report));

		for (DBObject obj : output.results()) {

			DBObject idField = (DBObject) obj.get("_id"); 
			DateTime time = MongoAggregationHelper.generateAggregationTime(idField, report.getAggregationGranule());
			double volume = Double.parseDouble(obj.get("storageStatusAvgVolume").toString());
			String filter = idField.get(report.getDimension()).toString();

			AggregatedResult current = new AggregatedResult(time, Metric.AVG, volume, filter);
			out.add(current);
		}

		Collections.sort((List<AggregatedResult>) out);
		return out;

	}


	//	private void checkReportFields(Report report){
	//
	//	}
}