

package org.gcube.accounting.common;

import java.util.Calendar;

public class TimeRange {

	private Calendar from;
	private Calendar to;

	/**
	 * The constructor guarantees that 'from' is always before 'to'.
	 * Furthermore, at least the 'left' end of the range must be defined
	 * 
	 * @param from
	 * @param to
	 */
	public TimeRange(Calendar from, Calendar to) {
		this.from = from;
		this.to = to;

		if (this.to.getTimeInMillis() == 0) {
			this.to = Calendar.getInstance();
		}

		if (from != null && to != null) {
			if (to.getTimeInMillis() < from.getTimeInMillis()) {
				Calendar aux = this.from;
				this.from = this.to;
				this.to = aux;
			}
		}

	}

	/**
	 * Getter for 'from'
	 * 
	 * @return
	 */
	public Calendar getFrom() {
		return this.from;
	}

	/**
	 * Getter for 'to'
	 * 
	 * @return
	 */
	public Calendar getTo() {
		return this.to;
	}

	/**
	 * Check if from and to are set
	 * 
	 * @return
	 */
	public boolean isClosed() {
		return this.from != null && this.to != null;
	}

	public boolean isUndefined() {
		return this.from == null && this.to == null;
	}

	/**
	 * Check whether this time range ends within the given time range
	 * 
	 * @param other
	 * @return
	 */
	public boolean endsWithin(TimeRange other) {
		if (this.to == null) {
			return false;
		}
		if (other.getFrom() == null && other.getTo() == null) {
			return false;
		}
		return this.to.after(other.getFrom()) && this.to.before(other.getTo());
	}

	/**
	 * Check whether this time range starts within the given time range
	 * 
	 * @param other
	 * @return
	 */
	public boolean startsWithin(TimeRange other) {
		if (this.to == null) {
			return false;
		}
		if (other.getFrom() == null && other.getTo() == null) {
			return false;
		}
		return this.to.after(other.getFrom()) && this.to.before(other.getTo());
	}

	/**
	 * Check whether this time range enrirely falls outside the the given time
	 * range
	 * 
	 * @param other
	 * @return
	 */
	public boolean isOutside(TimeRange otherRange) {
		if (this.isBefore(otherRange)) {
			return true;
		}
		if (this.isAfter(otherRange)) {
			return true;
		}
		return false;
	}

	/**
	 * Force this time range to be within the given time range
	 * 
	 * @param from
	 * @param to
	 */
	public void constrainTo(TimeRange otherRange) {
		if (this.isBefore(otherRange)) {
			return;
		}
		if (this.isAfter(otherRange)) {
			return;
		}
		if (this.isWithin(otherRange)) {
			return;
		}
		if (this.from.before(otherRange.getFrom())) {
			this.from = otherRange.getFrom();
		}
		if (this.to.after(otherRange.getTo())) {
			this.to = otherRange.getTo();
		}
	}

	/**
	 * Whether this range is entirely within 'range'
	 * 
	 * @param range
	 * @return
	 */
	public boolean isWithin(TimeRange range) {
		if (!this.isClosed() || !range.isClosed()) {
			return false;
		} else {
			return (this.from.after(range.getFrom()) && this.to.before(range
					.getTo()));
		}
	}

	/**
	 * Whether this range ends before 'range' starts
	 * 
	 * @param range
	 * @return
	 */
	public boolean isBefore(TimeRange range) {
		if (this.to == null) {
			return false;
		}
		if (range.getFrom() == null) {
			return false;
		}
		return this.to.before(range.getFrom());
	}

	/**
	 * Whether this range starts before 'range' ends
	 * 
	 * @param range
	 * @return
	 */
	public boolean isAfter(TimeRange range) {
		if (this.from == null) {
			return false;
		}
		if (range.getTo() == null) {
			return false;
		}
		return this.from.after(range.getTo());
	}

	/**
	 * The range extension, in milliseconds
	 * 
	 * @return
	 */
	public long getElapsed() {
		long start = (this.from != null ? this.from.getTimeInMillis() : 0);
		long end = (this.to != null ? this.to.getTimeInMillis() : Calendar
				.getInstance().getTimeInMillis());
		return (end - start);
	}

}
