/**
 * (c) 2014 FAO / UN (project: fi-security-server)
 */
package org.fao.fi.security.server.javax.filters.origin;

import java.io.IOException;
import java.util.Arrays;
import java.util.regex.Pattern;

import javax.annotation.PostConstruct;
import javax.annotation.Priority;
import javax.inject.Inject;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Priorities;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;

import org.fao.fi.security.server.javax.filters.AbstractRequestValidatorFilter;
import org.fao.fi.security.server.javax.filters.origin.configuration.RestrictedIPsConfiguration;
import org.fao.fi.security.server.javax.filters.origin.configuration.RestrictedIPsConstants;
import org.fao.fi.security.server.javax.filters.origin.support.IPRestrictedResource;
import org.fao.fi.security.server.javax.filters.origin.support.IPRestrictor;

/**
 * Place your class / interface description here.
 *
 * History:
 *
 * ------------- --------------- -----------------------
 * Date			 Author			 Comment
 * ------------- --------------- -----------------------
 * 30 Apr 2014   Fiorellato     Creation.
 *
 * @version 1.0
 * @since 30 Apr 2014
 */
@IPRestrictedResource @Priority(value=Priorities.AUTHENTICATION)
public class IPRestrictedResourceRequestValidatorFilter extends AbstractRequestValidatorFilter {
	@Context private HttpServletRequest _servletRequest;
	
	private RestrictedIPsConfiguration _configuration;
	private Pattern _validationPattern;

	@Inject private IPIdentifier _IPIdentifier;
	
	/**
	 * Class constructor
	 */
	@Inject public IPRestrictedResourceRequestValidatorFilter(@IPRestrictor RestrictedIPsConfiguration configuration) {
		this._configuration = configuration;
	}
	
	@PostConstruct 
	private void completeInitialization() {
		final String $THIS = this.getClass().getSimpleName() + "#" + this.hashCode();
		
		this._log.info("{} has been initialized with {} and {} as IP identifier", $THIS, this._configuration, this._IPIdentifier);
		
		this._validationPattern = this._configuration.getValidationPattern();
	}

	/* (non-Javadoc)
	 * @see org.fao.fi.security.server.javax.filters.AbstractRequestValidatorFilter#getSecurityScheme()
	 */
	@Override
	protected String getSecurityScheme() {
		return RestrictedIPsConstants.RESTRICTED_IP_SECURITY_TYPE_HEADER;
	}
	
	@Override
	public void filter(ContainerRequestContext requestContext) throws IOException {
		try {
			String host = this._servletRequest.getRemoteHost();
			String xff = requestContext.getHeaderString("X-FORWARDED-FOR");
			
			this._log.debug("Intercepted a {} request for {} coming from host {} [ XFF: {} ]", requestContext.getMethod(), requestContext.getUriInfo().getAbsolutePath(), host, xff == null ? "NOT SET" : xff);

			String[] IPs = this._IPIdentifier.identify(this._servletRequest);

			this._log.info("Request IPs being checked are: {}", Arrays.asList(IPs));
			
			boolean allow = IPs != null && IPs.length > 0;

			if(!allow)
				this._log.warn("Unable to identify IPs from incoming request, based on remote host {}, XFF {} and IP identifier of type {}", host, xff == null ? "[ NOT SET ]" : xff, this._IPIdentifier.getClass().getSimpleName());
			
			for(String IP : IPs) {
				this._log.info("Evaluating validity of request coming from {}", IP);

				this._log.info("Currently allowed (based on previous identified IPs in request and XFF): {}", allow);
				
				allow &= this._validationPattern.matcher(IP).matches();
				
				this._log.info("Allowed (after applying filtering on IP {}): {}", IP, allow);
			}
			
			if(!allow) {
				requestContext.abortWith(
					errorResponse(Response.Status.FORBIDDEN, "You are not authorized to access this resource")
				);
				
				this._log.warn("Blocked a {} request for {} coming from host {} [ XFF: {} ]", requestContext.getMethod(), requestContext.getUriInfo().getAbsolutePath(), host, xff == null ? "NOT SET" : xff);
			}
		} catch(Throwable t) {
			this._log.error("Unexpected {} caught: {}", t.getClass(), t.getMessage(), t);
			
			requestContext.abortWith(
				errorResponse(Response.Status.INTERNAL_SERVER_ERROR, t.getMessage())
			);
		}
	}
}